//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/ItemWithMaterial.cpp
//! @brief     Implements class ItemWithMaterial
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/ItemWithMaterial.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Sample/MaterialItem.h"
#include "GUI/Model/Sample/MaterialModel.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString MaterialId("MaterialId");

} // namespace Tag
} // namespace

ItemWithMaterial::ItemWithMaterial(const MaterialModel* materialModel)
    : m_materialModel(materialModel)
{
    ASSERT(m_materialModel);
}

void ItemWithMaterial::setMaterial(const MaterialItem* materialItem)
{
    m_materialIdentifier = materialItem->identifier();
}

void ItemWithMaterial::setMaterial(const QString& materialIdentifier)
{
    m_materialIdentifier = materialIdentifier;
}


QColor ItemWithMaterial::materialColor() const
{
    ASSERT(materialItem());
    return materialItem()->color();
}

QString ItemWithMaterial::materialName() const
{
    ASSERT(materialItem());
    return materialItem()->matItemName();
}

QString ItemWithMaterial::materialIdentifier() const
{
    return m_materialIdentifier;
}

MaterialItem* ItemWithMaterial::materialItem() const
{
    if (materialIdentifier().isEmpty())
        return nullptr;

    return m_materialModel->materialItemFromIdentifier(materialIdentifier());
}

void ItemWithMaterial::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // material id
    w->writeStartElement(Tag::MaterialId);
    XML::writeAttribute(w, XML::Attrib::value, m_materialIdentifier);
    w->writeEndElement();
}

void ItemWithMaterial::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // material id
        if (tag == Tag::MaterialId) {
            XML::readAttribute(r, XML::Attrib::value, &m_materialIdentifier);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}
