/**
 * Copyright (C) 2009-2013 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package footleg.cavesurvey.data.model;

/**
 * Represents a leg in a cave survey. A leg is a set of measurements which define the 
 * location of a point in the cave relative to another point (a survey station). The
 * second point defined by the leg can be another survey station or may just be a point
 * in the cave on the wall or used to measure the position of a significant feature.
 
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2013.08.29                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 * 
 * @to.do
 * TODO Add support for diving data
 */
public class SurveyLeg implements Comparable<SurveyLeg> {
	private SurveyStation fromStn;
	private SurveyStation toStn;
	private double length = -1;
	private double compass = -1;
	private double clino = -99;
	private String comment = "";
	
	//LRUD applies to fromStn
	private double left;
	private double right;
	private double up;
	private double down;
	
	//Leg flags
	private boolean splay = false;
	private boolean duplicate = false;
	private boolean surface = false;
	
	/**
	 * Support sorting by from station name, then to station name
	 */
	public int compareTo(SurveyLeg anotherLeg) {
		SurveyLeg leg = (SurveyLeg)anotherLeg;
		int fromTest = this.fromStn.getId() - leg.getFromStn().getId();
		if ( fromTest == 0 ) {
			//Same from station, so use toStn to sort
			return this.toStn.getId() - leg.getToStn().getId();
		}
		else {
			return fromTest;
		}
	}
	
	/**
	 * @return A copy of the survey leg
	 */
	public SurveyLeg clone() {
		SurveyLeg clone = new SurveyLeg();

		if (fromStn != null){
			clone.fromStn = fromStn.clone();
		}
		if (toStn != null){
			clone.toStn = toStn.clone();
		}
		clone.length = length;
		clone.compass = compass;
		clone.clino = clino;
		clone.comment = comment;
		clone.left = left;
		clone.right = right;
		clone.up = up;
		clone.down = down;
		clone.splay = splay;
		clone.duplicate = duplicate;
		clone.surface = surface;
		
		return clone;
	}
	
	/**
	 * Reverses the direction of a survey leg
	 */
	public void reverseDirection() {
		//Reverse leg direction
		SurveyStation stn = getFromStn();
		setFromStn( getToStn() );
		setToStn( stn );
		double compass = getCompass();
		if ( compass > 180 ) {
			compass -= 180;
		}
		else {
			compass += 180;
		}
		setCompass( compass );
		setClino( -getClino() );
	}
	
	/**
	 * Calculates the horizontal length of the leg
	 * @return calculated horizontal length
	 */
	public double getHorizontalLength() {
		double hLength = Math.abs( length * Math.cos(Math.PI * clino / 180) );
		return hLength;
	}
	
	/**
	 * Calculates the vertical height of the leg
	 * @return calculated vertical height
	 */
	public double getVerticalLength() {
		double vLength = Math.abs( length * Math.sin(Math.PI * clino / 180) );
		return vLength;
	}

	//Getters and Setters
	public SurveyStation getFromStn() {
		return fromStn;
	}
	public void setFromStn(SurveyStation fromStn) {
		this.fromStn = fromStn;
	}
	public SurveyStation getToStn() {
		return toStn;
	}
	public void setToStn(SurveyStation toStn) {
		this.toStn = toStn;
	}
	public double getLength() {
		return length;
	}
	public void setLength(double length) {
		this.length = length;
	}
	public double getCompass() {
		return compass;
	}
	public void setCompass(double compass) {
		this.compass = compass;
	}
	public double getClino() {
		return clino;
	}
	public void setClino(double clino) {
		this.clino = clino;
	}
	public double getLeft() {
		return left;
	}
	public void setLeft(double left) {
		this.left = left;
	}
	public double getRight() {
		return right;
	}
	public void setRight(double right) {
		this.right = right;
	}
	public double getUp() {
		return up;
	}
	public void setUp(double up) {
		this.up = up;
	}
	public double getDown() {
		return down;
	}
	public void setDown(double down) {
		this.down = down;
	}

	public String getComment() {
		return comment;
	}

	public void setComment(String comment) {
		this.comment = comment;
	}

	public boolean isSplay() {
		return splay;
	}

	public void setSplay(boolean splay) {
		this.splay = splay;
	}

	public boolean isDuplicate() {
		return duplicate;
	}

	public void setDuplicate(boolean duplicate) {
		this.duplicate = duplicate;
	}

	public boolean isSurface() {
		return surface;
	}

	public void setSurface(boolean surface) {
		this.surface = surface;
	}
}
