/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _BSD_SOURCE /* strdup() */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <stddef.h>

#include "gettext.h"
#include "cdw_list_display.h"
#include "cdw_text_file_viewer.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_window.h"
#include "cdw_debug.h"



/**
   \file cdw_text_file_viewer.c
   \brief Implementation of text file viewer widget for cdw

   File implements viewer of text files, that can be used e.g.
   as viewer of log file.
*/


/* maximal number of characters read in one line from file; lines in file
   longer than this limit will be truncated in this viewer; value does
   not include ending '\0' */
#define READ_BUFFER_SIZE 1024

/* maximal number of lines that can be read from file; I'm using this limit for performance reasons */
#define N_LINES_READ_MAX 100000


static void cdw_text_file_viewer_print_line(void *display, void *data, size_t row, size_t h_offset, bool isreverse);
static ssize_t cdw_text_file_viewer_read_file(FILE *file, cdw_dll_item_t **list);
static cdw_rv_t cdw_text_file_viewer_dealloc_lines_from_list(cdw_dll_item_t *list);





/**
   \brief Read content of the file to given list

   Function reads content of the \p file, line by line, and copies every line
   into \p list.

   There are two limits for this function:
   \li it can't read more than READ_BUFFER_SIZE chars at a time (this may
   lead to line truncations or some strange results in viewer's window)
   \li it can't read more than N_LINES_READ_MAX lines of file

   You can increase these two limits (independently) as you wish, memory and
   speed seem to be the only limiting factors (I haven't tested the limits
   very hard, currently they are set as 1024 and 10000, respectively).

   \p file must be already open for reading, \p list may be non-empty, the
   function will simply append more lines. Note that if any error occurs in
   the middle of reading file, the function does not rewind changes made to
   the \p list - it may be half-filled, possibly with incorrect data.

   Function uses cdw_dll_append_non_unique() so don't worry if repeating
   lines of text from file will be stored correctly on list, or about
   predicate for append() function.

   \param file - file to read from
   \param list - place where to copy lines read from file

   \return non-negative number of lines read from file (number of items appended to \p list)
   \return -1 on errors
*/
ssize_t cdw_text_file_viewer_read_file(FILE *file, cdw_dll_item_t **list)
{
	rewind(file);

	char read_buffer[READ_BUFFER_SIZE + 1];
	ssize_t n_lines = 0;
	bool success = true;

	while (n_lines < N_LINES_READ_MAX) {
		/* fgets() puts '\0' at the end of read buffer */
		char *c = fgets(read_buffer, READ_BUFFER_SIZE + 1, file);
		if (!c) {
			success = true;
			break;
		}
		char *data = strdup(read_buffer);
		if (!data) {
			success = false;
			break;
		}

		cdw_dll_append_non_unique(list, (void *) data);
		n_lines++;
	}

	if (success) {
		return n_lines;
	} else {
		return -1;
	}
}





/**
   \brief Print one line of text in given display

   Function prints one line of text from \p data in given \p row of window
   associated with given \p display. Function can handle non-zero values
   of \p h_offset: displayed line of text will be shifted to the left
   accordingly.

   This is the "item display function" of text file viewer module. You should
   assign it to display->display_item after creating display for text file
   viewer.

   \param displ - display to use as a place to print a line
   \param data - data to be printed in given display
   \param row - specific row number of display's window, in which to print a line
   \param h_offset - horizontal offset, with which to print a line
   \param isreverse - should the function use reverse display attributes to display the line?
*/
void cdw_text_file_viewer_print_line(void *displ, void *data, size_t row, size_t h_offset, bool isreverse)
{
	CDW_LIST_DISPLAY *display = (CDW_LIST_DISPLAY *) displ;
	char *buffer = (char *) data;

	if (!buffer) {
		return;
	}

	#if 0

	/* this code expands tabs into spaces to avoid "jumping tabs" in window */
	size_t t = 0;
	bool end = false; /* did we got to end of read_buffer? */
	for (t = 0; t < strlen(buffer); t++) {
		if (buffer[t] == '\t') {
			char tmp_line[READ_BUFFER_SIZE];
			strncpy(tmp_line, buffer, t);

			size_t j = 0;
			/* 8 is 8 spaces for a TAB character */
			for (j = 0; j < 8; j++) {
				if (t + j < READ_BUFFER_SIZE) {
					tmp_line[t + j] = ' ';
				} else {
					end = true;
					break;
				}
			}

			if (end) {
				break;
			}

			/* this puts content of 'read_buffer' that is
			   after '\t' into place after eight spaces */
			strncpy(tmp_line + t + 8, buffer + t + 1, READ_BUFFER_SIZE - t - 8);

			strncpy(buffer, tmp_line, READ_BUFFER_SIZE);
			buffer[READ_BUFFER_SIZE] = '\0';
		}
	}
	#endif

	long unsigned int attr = 0;
	if (isreverse) {
		attr = A_REVERSE | COLOR_PAIR(display->colors);
	} else {
		attr = A_NORMAL | COLOR_PAIR(display->colors);
	}

	(void) wattrset(display->subwindow, attr);

	if (strlen(buffer) >= h_offset) {
		mvwprintw(display->subwindow, (int) row, 0, "%s", (char *) buffer + h_offset);
	} else {
		mvwprintw(display->subwindow, (int) row, 0, " ");
	}

	return;
}





/**
   \brief Open given file, display its content in window, close file

   Function opens file specified by \p fullpath, calls cdw_text_file_viewer()
   to create a viewer, display content of file and handle keys from user, and
   then closes file and returns.

   \p file_path may not be NULL nor empty, \p title may be empty string or NULL.

   \param fullpath - full path to file to be opened
   \param title - string used as title of display window

   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_text_file_viewer_with_fullpath(const char *fullpath, const char *title)
{
	cdw_assert (fullpath != (char *) NULL, "file path is null\n");
	cdw_assert (strlen(fullpath), "file path len is zero\n");

	FILE *file = fopen(fullpath, "r");
	if (!file) {
		return CDW_ERROR;
	}

	cdw_rv_t crv = cdw_text_file_viewer(file, title);

	fclose(file);
	file = (FILE *) NULL;

	return crv;
}





/**
   \brief Display content of file using list display widget

   Function creates instance of list display widget, calls
   cdw_text_file_viewer_read_file() to read content of \p file to list
   associated with the display, and calls driver function for the display.
   When user closes display window, function cleans some things up.

   \p file must be already open for reading. Function does not close it.
   \p title is a title of window displaying content of file. It may be
   empty string or NULL.

   \param file - open file that caller want to display
   \param title - string used as title of display window

   \return CDW_OK on success
   \return CDW_GEN_ERROR on errors
*/
cdw_rv_t cdw_text_file_viewer(FILE *file, const char *title)
{
	cdw_assert (COLS > 6, "ERROR: COLS is %d, should be more than 6\n", COLS);
	cdw_assert (LINES > 4, "ERROR: LINES is %d, should be more than 4\n", LINES);

	/* these are parameters of parent, stand-alone window */
	int n_cols = COLS - 2;
	int n_lines = LINES - 2;
	int begin_y = 1;
	int begin_x = 1;

	WINDOW *window = newwin(n_lines, n_cols, begin_y, begin_x);
	if (!window) {
		cdw_vdm ("ERROR: failed to create window for text file viewer\n");
		return CDW_ERROR;
	}

	CDW_LIST_DISPLAY *display = cdw_list_display_new(window, n_lines - 2, n_cols - 2, 1, 1, CDW_COLORS_DIALOG);
	if (!display) {
		delwin (window);
		window = (WINDOW *) NULL;

		cdw_vdm ("ERROR: failed to create display for text file viewer\n");
		return CDW_ERROR;
	}

	display->display_item = cdw_text_file_viewer_print_line;
	display->display_format = CDW_LIST_DISPLAY_FORMAT_LONG;

	/* 2TRANS: this is help text displayed at the bottom of text file
	   viewer window: it explains which keys can be used */
	cdw_window_add_strings(display->window, title, _("Use ESC to close, use arrows, Home and End to move"));

	ssize_t n = cdw_text_file_viewer_read_file(file, &(display->list));
	if (n != -1) {
		display->n_items = (size_t) n;
		cdw_list_display_scroll_to(display, 0, 0, false);
		cdw_list_display_refresh(display);

		/* this function will control the display until user pressed ESCAPE */
		cdw_list_display_driver(display);
	}

	/* TODO: display should have pointer to function that knows how to free list content */
	cdw_text_file_viewer_dealloc_lines_from_list(display->list);
	/* cdw_list_display_delete() also deallocates list data structure */
	cdw_list_display_delete(&display);

	delwin(window);
	window = (WINDOW *) NULL;

	cdw_main_window_wrefresh_part(n_lines, n_cols, begin_y, begin_x);
	if (n == -1) {
		cdw_vdm ("ERROR: problems occurred when reading file\n");
		/* there will be no error message for user here, it should
		   be handled by code calling text file viewer functions */
		return CDW_ERROR;
	} else if (n == 0) {
		/* not necessarily an error, but it's worth informing
		   in debug output */
		cdw_vdm ("WARNING: zero lines read from file\n");
		return CDW_OK;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Deallocate data stored on given list

   Function deallocates date stored on given list. This function is
   specialized for data type used by text file viewer, which is ordinary
   "char *" C string. Call this function before destroying text file display.

   \param list - list from which to reallocate lines of text

   \return CDW_OK
*/
cdw_rv_t cdw_text_file_viewer_dealloc_lines_from_list(cdw_dll_item_t *list)
{
	for (cdw_dll_item_t *item = list; item; item = item->next) {
		free((char *) item->data);
		item->data = (char *) NULL;
	}

	return CDW_OK;
}
