!--------------------------------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations                              !
!   Copyright 2000-2024 CP2K developers group <https://cp2k.org>                                   !
!                                                                                                  !
!   SPDX-License-Identifier: GPL-2.0-or-later                                                      !
!--------------------------------------------------------------------------------------------------!

! **************************************************************************************************
!> \brief Utilities for string manipulations
!> \par History
!>      Adapted compress and uppercase for use in CP2K (JGH)
!>      string_to_integer and integer_to_string added (06.02.2001, MK)
!>      Cleaned (04.01.2004,MK)
!> \author MK & JGH
! **************************************************************************************************
MODULE string_utilities

   USE ISO_C_BINDING,                   ONLY: C_CHAR,&
                                              C_NULL_CHAR
   USE kinds,                           ONLY: default_blank_character

   IMPLICIT NONE

   PRIVATE

   CHARACTER(LEN=1), PARAMETER :: backslash = '\\'
   CHARACTER(LEN=1), PARAMETER :: star = '*'
   CHARACTER(LEN=1), PARAMETER :: question = '?'
   CHARACTER(LEN=1), PARAMETER :: newline = ACHAR(10)

   PUBLIC :: ascii_to_string, &
             compress, &
             integer_to_string, &
             is_whitespace, &
             remove_word, &
             s2a, a2s, &
             str_comp, &
             string_to_ascii, &
             strip_control_codes, &
             substitute_special_xml_tokens, &
             typo_match, &
             uppercase, &
             xstring, &
             strlcpy_c2f

   PUBLIC :: newline

   INTERFACE s2a
      MODULE PROCEDURE s2a_1, s2a_2, s2a_3, s2a_4, s2a_5, s2a_6, s2a_7, s2a_8, s2a_9, &
         s2a_10, s2a_11, s2a_12, s2a_13, s2a_14, s2a_15, s2a_16, s2a_17, s2a_18, s2a_19, &
         s2a_20, s2a_21, s2a_22, s2a_23, s2a_24, s2a_25, s2a_26, s2a_27, s2a_28, s2a_29, &
         s2a_30, s2a_31, s2a_32, s2a_33, s2a_34, s2a_35, s2a_36, s2a_37, s2a_38, s2a_39, &
         s2a_40, s2a_41, s2a_42, s2a_43, s2a_44, s2a_45, s2a_46, s2a_47, s2a_48, s2a_49, &
         s2a_50, s2a_51, s2a_52, s2a_53, s2a_54, s2a_55, s2a_56, s2a_57, s2a_58, s2a_59 ! should be clear how to add more
   END INTERFACE

CONTAINS

! **************************************************************************************************
!> \brief returns a non-zero positive value if typo_string equals string apart from a few typos.
!>     It is case sensitive, apart from typos.
!> \param string ...
!> \param typo_string ...
!> \return ...
!> \par History
!>      02.2006 created [Joost VandeVondele]
!>      09.2020 switch to Levenshtein distance [Ole Schuett]
! **************************************************************************************************
   ELEMENTAL FUNCTION typo_match(string, typo_string) RESULT(match)
      CHARACTER(LEN=*), INTENT(IN)                       :: string, typo_string
      INTEGER                                            :: match

      match = MAX(0, 100 - levenshtein_distance(string, typo_string))

   END FUNCTION typo_match

! **************************************************************************************************
!> \brief Compute the Levenshtein edit distance between the two given strings.
!> \param s source string
!> \param t target string
!> \return edit distance
!> \author Ole Schuett
! **************************************************************************************************
   PURE FUNCTION levenshtein_distance(s, t) RESULT(distance)

      CHARACTER(LEN=*), INTENT(IN)                       :: s, t
      INTEGER                                            :: distance

      INTEGER                                            :: i, j, m, n, substitution_cost
      INTEGER, ALLOCATABLE, DIMENSION(:, :)              :: d

      m = LEN_TRIM(s)
      n = LEN_TRIM(t)

      ! distance between the first i characters of s and the first j characters of t
      ALLOCATE (d(0:m, 0:n))
      d(:, :) = 0

      ! source prefixes can be transformed into empty string by dropping all characters
      DO i = 1, m
         d(i, 0) = i
      END DO

      ! target prefixes can be reached from empty source prefix by inserting every character
      DO j = 1, n
         d(0, j) = j
      END DO

      DO j = 1, n
         DO i = 1, m
            IF (s(i:i) == t(j:j)) THEN
               substitution_cost = 0
            ELSE
               substitution_cost = 1
            END IF
            d(i, j) = MIN(d(i - 1, j) + 1, &                    ! deletion
                          d(i, j - 1) + 1, &                    ! insertion
                          d(i - 1, j - 1) + substitution_cost)  ! substitution
         END DO
      END DO

      distance = d(m, n)

   END FUNCTION levenshtein_distance

! **************************************************************************************************
!> \brief Converts a character-array into a string
!> \param array ...
!> \return ...
!> \par History
!>      10.2013 created [Ole Schuett]
! **************************************************************************************************
   PURE FUNCTION a2s(array) RESULT(string)
      CHARACTER, DIMENSION(:), INTENT(IN)                :: array
      CHARACTER(LEN=SIZE(array))                         :: string

      INTEGER                                            :: i

      DO i = 1, SIZE(array)
         string(i:i) = array(i)
      END DO
   END FUNCTION a2s

! **************************************************************************************************
!> \brief converts a bunch of strings of different length to an array of
!>        strings of the same length
!> \param s1 ...
!> \return ...
!> \par History
!>      11.2004 created [Joost VandeVondele ]
!> \note
!>     can be used instead of the illegal (/"12","1234"/) generating
!>     s2a("12","1234").EQ.(/"12  ","1234"/)
! **************************************************************************************************
   PURE FUNCTION s2a_1(s1) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1
      CHARACTER(LEN=1000), DIMENSION(1)                  :: a

      a(1) = s1
   END FUNCTION s2a_1
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_2(s1, s2) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2
      CHARACTER(LEN=1000), DIMENSION(2)                  :: a

      a(1) = s1; a(2) = s2
   END FUNCTION s2a_2
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_3(s1, s2, s3) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3
      CHARACTER(LEN=1000), DIMENSION(3)                  :: a

      a(1) = s1; a(2) = s2; a(3) = s3
   END FUNCTION s2a_3
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_4(s1, s2, s3, s4) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4
      CHARACTER(LEN=1000), DIMENSION(4)                  :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4
   END FUNCTION s2a_4
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_5(s1, s2, s3, s4, s5) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5
      CHARACTER(LEN=1000), DIMENSION(5)                  :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
   END FUNCTION s2a_5
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_6(s1, s2, s3, s4, s5, s6) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6
      CHARACTER(LEN=1000), DIMENSION(6)                  :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6
   END FUNCTION s2a_6
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_7(s1, s2, s3, s4, s5, s6, s7) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7
      CHARACTER(LEN=1000), DIMENSION(7)                  :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
   END FUNCTION s2a_7
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_8(s1, s2, s3, s4, s5, s6, s7, s8) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8
      CHARACTER(LEN=1000), DIMENSION(8)                  :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8
   END FUNCTION s2a_8
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_9(s1, s2, s3, s4, s5, s6, s7, s8, s9) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9
      CHARACTER(LEN=1000), DIMENSION(9)                  :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9
   END FUNCTION s2a_9
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_10(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10
      CHARACTER(LEN=1000), DIMENSION(10)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10
   END FUNCTION s2a_10
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_11(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11
      CHARACTER(LEN=1000), DIMENSION(11)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11
   END FUNCTION s2a_11
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_12(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12
      CHARACTER(LEN=1000), DIMENSION(12)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12
   END FUNCTION s2a_12
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_13(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13
      CHARACTER(LEN=1000), DIMENSION(13)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13
   END FUNCTION s2a_13
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_14(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14
      CHARACTER(LEN=1000), DIMENSION(14)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
   END FUNCTION s2a_14
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_15(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15
      CHARACTER(LEN=1000), DIMENSION(15)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15
   END FUNCTION s2a_15
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_16(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16
      CHARACTER(LEN=1000), DIMENSION(16)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16
   END FUNCTION s2a_16
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_17(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17
      CHARACTER(LEN=1000), DIMENSION(17)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17
   END FUNCTION s2a_17
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_18(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, s18
      CHARACTER(LEN=1000), DIMENSION(18)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18
   END FUNCTION s2a_18
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_19(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19
      CHARACTER(LEN=1000), DIMENSION(19)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19
   END FUNCTION s2a_19
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_20(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20
      CHARACTER(LEN=1000), DIMENSION(20)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
   END FUNCTION s2a_20
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_21(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21
      CHARACTER(LEN=1000), DIMENSION(21)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21
   END FUNCTION s2a_21
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_22(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22
      CHARACTER(LEN=1000), DIMENSION(22)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22
   END FUNCTION s2a_22
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_23(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23
      CHARACTER(LEN=1000), DIMENSION(23)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23
   END FUNCTION s2a_23
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_24(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24
      CHARACTER(LEN=1000), DIMENSION(24)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24
   END FUNCTION s2a_24
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_25(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, s25
      CHARACTER(LEN=1000), DIMENSION(25)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
   END FUNCTION s2a_25
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_26(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26
      CHARACTER(LEN=1000), DIMENSION(26)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
   END FUNCTION s2a_26
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_27(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27
      CHARACTER(LEN=1000), DIMENSION(27)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27
   END FUNCTION s2a_27
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_28(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27, s28
      CHARACTER(LEN=1000), DIMENSION(28)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28
   END FUNCTION s2a_28
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_29(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27, s28, s29
      CHARACTER(LEN=1000), DIMENSION(29)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29
   END FUNCTION s2a_29
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_30(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27, s28, s29, s30
      CHARACTER(LEN=1000), DIMENSION(30)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
   END FUNCTION s2a_30
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_31(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27, s28, s29, s30, s31
      CHARACTER(LEN=1000), DIMENSION(31)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31
   END FUNCTION s2a_31
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_32(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27, s28, s29, s30, s31, s32
      CHARACTER(LEN=1000), DIMENSION(32)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31; a(32) = s32
   END FUNCTION s2a_32
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_33(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
                        s33) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27, s28, s29, s30, s31, &
                                                            s32, s33
      CHARACTER(LEN=1000), DIMENSION(33)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31; a(32) = s32
      a(33) = s33
   END FUNCTION s2a_33
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_34(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
                        s33, s34) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27, s28, s29, s30, s31, &
                                                            s32, s33, s34
      CHARACTER(LEN=1000), DIMENSION(34)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31; a(32) = s32
      a(33) = s33; a(34) = s34
   END FUNCTION s2a_34
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_35(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
                        s33, s34, s35) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN)                       :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                                                            s11, s12, s13, s14, s15, s16, s17, &
                                                            s18, s19, s20, s21, s22, s23, s24, &
                                                            s25, s26, s27, s28, s29, s30, s31, &
                                                            s32, s33, s34, s35
      CHARACTER(LEN=1000), DIMENSION(35)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31; a(32) = s32
      a(33) = s33; a(34) = s34; a(35) = s35
   END FUNCTION s2a_35
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_36(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
                        s33, s34, s35, s36) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36
      CHARACTER(LEN=1000), DIMENSION(36)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31; a(32) = s32
      a(33) = s33; a(34) = s34; a(35) = s35; a(36) = s36
   END FUNCTION s2a_36
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_37(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
                        s33, s34, s35, s36, s37) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37
      CHARACTER(LEN=1000), DIMENSION(37)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31; a(32) = s32
      a(33) = s33; a(34) = s34; a(35) = s35; a(36) = s36; a(37) = s37
   END FUNCTION s2a_37
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_38(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
                        s33, s34, s35, s36, s37, s38) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38
      CHARACTER(LEN=1000), DIMENSION(38)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31; a(32) = s32
      a(33) = s33; a(34) = s34; a(35) = s35; a(36) = s36; a(37) = s37; a(38) = s38
   END FUNCTION s2a_38
   ! **************************************************************************************************
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_39(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, &
                        s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
                        s33, s34, s35, s36, s37, s38, s39) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39
      CHARACTER(LEN=1000), DIMENSION(39)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5; a(6) = s6; a(7) = s7
      a(8) = s8; a(9) = s9; a(10) = s10; a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14
      a(15) = s15; a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25; a(26) = s26
      a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30; a(31) = s31; a(32) = s32
      a(33) = s33; a(34) = s34; a(35) = s35; a(36) = s36; a(37) = s37; a(38) = s38
      a(39) = s39
   END FUNCTION s2a_39
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_40(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40
      CHARACTER(LEN=1000), DIMENSION(40)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
   END FUNCTION s2a_40
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_41(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41
      CHARACTER(LEN=1000), DIMENSION(41)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41
   END FUNCTION s2a_41
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_42(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42
      CHARACTER(LEN=1000), DIMENSION(42)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42
   END FUNCTION s2a_42
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_43(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43
      CHARACTER(LEN=1000), DIMENSION(43)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43
   END FUNCTION s2a_43
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_44(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44
      CHARACTER(LEN=1000), DIMENSION(44)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44
   END FUNCTION s2a_44
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_45(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45
      CHARACTER(LEN=1000), DIMENSION(45)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
   END FUNCTION s2a_45
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_46(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46
      CHARACTER(LEN=1000), DIMENSION(46)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46
   END FUNCTION s2a_46
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_47(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47
      CHARACTER(LEN=1000), DIMENSION(47)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47
   END FUNCTION s2a_47
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_48(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48
      CHARACTER(LEN=1000), DIMENSION(48)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48
   END FUNCTION s2a_48
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_49(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49
      CHARACTER(LEN=1000), DIMENSION(49)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49
   END FUNCTION s2a_49
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_50(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50
      CHARACTER(LEN=1000), DIMENSION(50)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
   END FUNCTION s2a_50
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_51(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51
      CHARACTER(LEN=1000), DIMENSION(51)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51
   END FUNCTION s2a_51
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_52(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52
      CHARACTER(LEN=1000), DIMENSION(52)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52
   END FUNCTION s2a_52
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \param s53 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_53(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52, s53) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52, s53
      CHARACTER(LEN=1000), DIMENSION(53)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52; a(53) = s53
   END FUNCTION s2a_53
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \param s53 ...
!> \param s54 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_54(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52, s53, s54) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52, s53, s54
      CHARACTER(LEN=1000), DIMENSION(54)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52; a(53) = s53; a(54) = s54
   END FUNCTION s2a_54
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \param s53 ...
!> \param s54 ...
!> \param s55 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_55(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52, s53, s54, s55) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52, s53, s54, s55
      CHARACTER(LEN=1000), DIMENSION(55)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52; a(53) = s53; a(54) = s54; a(55) = s55
   END FUNCTION s2a_55
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \param s53 ...
!> \param s54 ...
!> \param s55 ...
!> \param s56 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_56(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52, s53, s54, s55, s56) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52, s53, s54, s55, s56
      CHARACTER(LEN=1000), DIMENSION(56)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52; a(53) = s53; a(54) = s54; a(55) = s55
      a(56) = s56
   END FUNCTION s2a_56
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \param s53 ...
!> \param s54 ...
!> \param s55 ...
!> \param s56 ...
!> \param s57 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_57(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52, s53, s54, s55, s56, s57) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52, s53, s54, s55, s56, s57
      CHARACTER(LEN=1000), DIMENSION(57)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52; a(53) = s53; a(54) = s54; a(55) = s55
      a(56) = s56; a(57) = s57
   END FUNCTION s2a_57
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \param s53 ...
!> \param s54 ...
!> \param s55 ...
!> \param s56 ...
!> \param s57 ...
!> \param s58 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_58(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52, s53, s54, s55, s56, s57, s58) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52, s53, s54, s55, s56, s57, s58
      CHARACTER(LEN=1000), DIMENSION(58)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52; a(53) = s53; a(54) = s54; a(55) = s55
      a(56) = s56; a(57) = s57; a(58) = s58
   END FUNCTION s2a_58
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \param s53 ...
!> \param s54 ...
!> \param s55 ...
!> \param s56 ...
!> \param s57 ...
!> \param s58 ...
!> \param s59 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_59(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52, s53, s54, s55, s56, s57, s58, s59) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52, s53, s54, s55, s56, s57, s58, s59
      CHARACTER(LEN=1000), DIMENSION(59)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52; a(53) = s53; a(54) = s54; a(55) = s55
      a(56) = s56; a(57) = s57; a(58) = s58; a(59) = s59
   END FUNCTION s2a_59
! **************************************************************************************************
!> \brief ...
!> \param s1 ...
!> \param s2 ...
!> \param s3 ...
!> \param s4 ...
!> \param s5 ...
!> \param s6 ...
!> \param s7 ...
!> \param s8 ...
!> \param s9 ...
!> \param s10 ...
!> \param s11 ...
!> \param s12 ...
!> \param s13 ...
!> \param s14 ...
!> \param s15 ...
!> \param s16 ...
!> \param s17 ...
!> \param s18 ...
!> \param s19 ...
!> \param s20 ...
!> \param s21 ...
!> \param s22 ...
!> \param s23 ...
!> \param s24 ...
!> \param s25 ...
!> \param s26 ...
!> \param s27 ...
!> \param s28 ...
!> \param s29 ...
!> \param s30 ...
!> \param s31 ...
!> \param s32 ...
!> \param s33 ...
!> \param s34 ...
!> \param s35 ...
!> \param s36 ...
!> \param s37 ...
!> \param s38 ...
!> \param s39 ...
!> \param s40 ...
!> \param s41 ...
!> \param s42 ...
!> \param s43 ...
!> \param s44 ...
!> \param s45 ...
!> \param s46 ...
!> \param s47 ...
!> \param s48 ...
!> \param s49 ...
!> \param s50 ...
!> \param s51 ...
!> \param s52 ...
!> \param s53 ...
!> \param s54 ...
!> \param s55 ...
!> \param s56 ...
!> \param s57 ...
!> \param s58 ...
!> \param s59 ...
!> \param s60 ...
!> \return ...
! **************************************************************************************************
   PURE FUNCTION s2a_60(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, &
                        s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, &
                        s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, &
                        s31, s32, s33, s34, s35, s36, s37, s38, s39, s40, &
                        s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
                        s51, s52, s53, s54, s55, s56, s57, s58, s59, s60) RESULT(a)
      CHARACTER(LEN=*), INTENT(IN) :: s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, &
         s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31, s32, &
         s33, s34, s35, s36, s37, s38, s39, s40, s41, s42, s43, s44, s45, s46, s47, s48, s49, s50, &
         s51, s52, s53, s54, s55, s56, s57, s58, s59, s60
      CHARACTER(LEN=1000), DIMENSION(60)                 :: a

      a(1) = s1; a(2) = s2; a(3) = s3; a(4) = s4; a(5) = s5
      a(6) = s6; a(7) = s7; a(8) = s8; a(9) = s9; a(10) = s10
      a(11) = s11; a(12) = s12; a(13) = s13; a(14) = s14; a(15) = s15
      a(16) = s16; a(17) = s17; a(18) = s18; a(19) = s19; a(20) = s20
      a(21) = s21; a(22) = s22; a(23) = s23; a(24) = s24; a(25) = s25
      a(26) = s26; a(27) = s27; a(28) = s28; a(29) = s29; a(30) = s30
      a(31) = s31; a(32) = s32; a(33) = s33; a(34) = s34; a(35) = s35
      a(36) = s36; a(37) = s37; a(38) = s38; a(39) = s39; a(40) = s40
      a(41) = s41; a(42) = s42; a(43) = s43; a(44) = s44; a(45) = s45
      a(46) = s46; a(47) = s47; a(48) = s48; a(49) = s49; a(50) = s50
      a(51) = s51; a(52) = s52; a(53) = s53; a(54) = s54; a(55) = s55
      a(56) = s56; a(57) = s57; a(58) = s58; a(59) = s59; a(60) = s60
   END FUNCTION s2a_60

! **************************************************************************************************
!> \brief Convert a sequence of integer numbers (ASCII code) to a string.
!>         Blanks are inserted for invalid ASCII code numbers.
!> \param nascii ...
!> \param string ...
!> \date    19.10.2000
!> \author  MK
!> \version 1.0
! **************************************************************************************************
   SUBROUTINE ascii_to_string(nascii, string)

      INTEGER, DIMENSION(:), INTENT(IN)                  :: nascii
      CHARACTER(LEN=*), INTENT(OUT)                      :: string

      INTEGER                                            :: i

      string = ""

      DO i = 1, MIN(LEN(string), SIZE(nascii))
         IF ((nascii(i) >= 0) .AND. (nascii(i) <= 127)) THEN
            string(i:i) = CHAR(nascii(i))
         ELSE
            string(i:i) = " "
         END IF
      END DO

   END SUBROUTINE ascii_to_string

! **************************************************************************************************
!> \brief   Eliminate multiple space characters in a string.
!>          If full is .TRUE., then all spaces are eliminated.
!> \param string ...
!> \param full ...
!> \date    23.06.1998
!> \author  MK
!> \version 1.0
! **************************************************************************************************
   SUBROUTINE compress(string, full)

      CHARACTER(LEN=*), INTENT(INOUT)                    :: string
      LOGICAL, INTENT(IN), OPTIONAL                      :: full

      CHARACTER                                          :: tmp
      INTEGER                                            :: i, z
      LOGICAL                                            :: remove_all

      IF (PRESENT(full)) THEN
         remove_all = full
      ELSE
         remove_all = .FALSE.
      END IF

      z = 1

      DO i = 1, LEN_TRIM(string)
         IF ((z == 1) .OR. remove_all) THEN
            IF (string(i:i) /= " ") THEN
               tmp = string(i:i)
               string(z:z) = tmp
               z = z + 1
            END IF
         ELSE
            IF ((string(i:i) /= " ") .OR. (string(z - 1:z - 1) /= " ")) THEN
               tmp = string(i:i)
               string(z:z) = tmp
               z = z + 1
            END IF
         END IF
      END DO

      string(z:) = ""

   END SUBROUTINE compress

! **************************************************************************************************
!> \brief   Converts an integer number to a string.
!>          The WRITE statement will return an error message, if the number of
!>          digits of the integer number is larger the than the length of the
!>          supplied string.
!> \param inumber ...
!> \param string ...
!> \date    05.01.2004
!> \author  MK
!> \version 1.0
! **************************************************************************************************
   SUBROUTINE integer_to_string(inumber, string)

      INTEGER, INTENT(IN)                                :: inumber
      CHARACTER(LEN=*), INTENT(OUT)                      :: string

      WRITE (UNIT=string, FMT='(I0)') inumber
   END SUBROUTINE integer_to_string

! **************************************************************************************************
!> \brief   Convert a string to sequence of integer numbers.
!> \param string ...
!> \param nascii ...
!> \date    19.10.2000
!> \author  MK
!> \version 1.0
! **************************************************************************************************
   SUBROUTINE string_to_ascii(string, nascii)

      CHARACTER(LEN=*), INTENT(IN)                       :: string
      INTEGER, DIMENSION(:), INTENT(OUT)                 :: nascii

      INTEGER                                            :: i

      nascii(:) = 0

      DO i = 1, MIN(LEN(string), SIZE(nascii))
         nascii(i) = ICHAR(string(i:i))
      END DO

   END SUBROUTINE string_to_ascii

! **************************************************************************************************
!> \brief   remove a word from a string (words are separated by white spaces)
!> \param string ...
!> \version 1.0
! **************************************************************************************************
   SUBROUTINE remove_word(string)
      CHARACTER(LEN=*), INTENT(INOUT)                    :: string

      INTEGER                                            :: i

      i = 1
      ! possibly clean white spaces
      DO WHILE (string(i:i) == " ")
         i = i + 1
      END DO
      ! now remove the word
      DO WHILE (string(i:i) /= " ")
         i = i + 1
      END DO
      string = string(i:)

   END SUBROUTINE remove_word

! **************************************************************************************************
!> \brief Substitutes the five predefined XML entities: &amp;, &lt;, &gt;, &apos;, and &quot;.
!> \param inp_string ...
!> \return ...
!> !> \author Ole Schuett
! **************************************************************************************************
   FUNCTION substitute_special_xml_tokens(inp_string) RESULT(out_string)

      CHARACTER(LEN=*), INTENT(IN)                       :: inp_string
      CHARACTER(LEN=2*LEN(inp_string))                   :: out_string

      INTEGER                                            :: i, j

      out_string = ""

      j = 1
      DO i = 1, LEN_TRIM(inp_string)
         SELECT CASE (inp_string(i:i))
         CASE ("<")
            out_string(j:j + 3) = "&lt;"
            j = j + 4
         CASE (">")
            out_string(j:j + 3) = "&gt;"
            j = j + 4
         CASE ("&")
            out_string(j:j + 4) = "&amp;"
            j = j + 5
         CASE ("'")
            out_string(j:j + 5) = "&apos;"
            j = j + 6
         CASE ('"')
            out_string(j:j + 5) = "&quot;"
            j = j + 6
         CASE DEFAULT
            out_string(j:j) = inp_string(i:i)
            j = j + 1
         END SELECT
      END DO

   END FUNCTION substitute_special_xml_tokens

! **************************************************************************************************
!> \brief   Convert all upper case characters in a string to lower case.
!> \param string ...
!> \date    14.08.2013
!> \author  Matthias Krack (MK)
!> \version 1.0
! **************************************************************************************************
   ELEMENTAL SUBROUTINE lowercase(string)
      CHARACTER(LEN=*), INTENT(INOUT)                    :: string

      INTEGER                                            :: i, iascii

      DO i = 1, LEN_TRIM(string)
         iascii = ICHAR(string(i:i))
         IF ((iascii >= 65) .AND. (iascii <= 90)) THEN
            string(i:i) = CHAR(iascii + 32)
         END IF
      END DO

   END SUBROUTINE lowercase

! **************************************************************************************************
!> \brief   Convert all lower case characters in a string to upper case.
!> \param string ...
!> \date    22.06.1998
!> \author  MK
!> \version 1.0
! **************************************************************************************************
   ELEMENTAL SUBROUTINE uppercase(string)
      CHARACTER(LEN=*), INTENT(INOUT)                    :: string

      INTEGER                                            :: i, iascii

      DO i = 1, LEN_TRIM(string)
         iascii = ICHAR(string(i:i))
         IF ((iascii >= 97) .AND. (iascii <= 122)) THEN
            string(i:i) = CHAR(iascii - 32)
         END IF
      END DO

   END SUBROUTINE uppercase

! **************************************************************************************************
!> \brief ...
!> \param string ...
!> \param ia ...
!> \param ib ...
! **************************************************************************************************
   ELEMENTAL SUBROUTINE xstring(string, ia, ib)

      CHARACTER(LEN=*), INTENT(IN)                       :: string
      INTEGER, INTENT(OUT)                               :: ia, ib

      ia = 1
      ib = LEN_TRIM(string)
      IF (ib > 0) THEN
         DO WHILE (string(ia:ia) == ' ')
            ia = ia + 1
         END DO
      END IF

   END SUBROUTINE xstring

! **************************************************************************************************
!> \brief ...
!> \param str1 ...
!> \param str2 ...
!> \return ...
! **************************************************************************************************
   ELEMENTAL FUNCTION str_comp(str1, str2) RESULT(equal)

      CHARACTER(LEN=*), INTENT(IN)                       :: str1, str2
      LOGICAL                                            :: equal

      INTEGER                                            :: i1, i2, j1, j2

      i1 = 0
      i2 = 0
      j1 = 0
      j2 = 0
      CALL xstring(str1, i1, i2)
      CALL xstring(str2, j1, j2)
      equal = (str1(i1:i2) == str2(j1:j2))
   END FUNCTION str_comp

! **************************************************************************************************
!> \brief returns .true. if the character passed is a whitespace char.
!> \param testchar ...
!> \return ...
!> \par History
!>      02.2008 created, AK
! **************************************************************************************************
   ELEMENTAL FUNCTION is_whitespace(testchar) RESULT(resval)
      CHARACTER(LEN=1), INTENT(IN)                       :: testchar
      LOGICAL                                            :: resval

      resval = .FALSE.
      IF (ANY(default_blank_character == testchar)) resval = .TRUE.
   END FUNCTION is_whitespace

! **************************************************************************************************
!> \brief Strip control codes and extended characters from a string, i.e. replace them with blanks
!> \param string ...
!> \date    13.12.2019
!> \author  MK
!> \version 1.0
! **************************************************************************************************
   ELEMENTAL SUBROUTINE strip_control_codes(string)

      CHARACTER(LEN=*), INTENT(INOUT)                    :: string

      INTEGER                                            :: i, iascii

      DO i = 1, LEN(string)
         iascii = ICHAR(string(i:i))
         IF ((iascii < 32) .OR. (iascii > 126)) string(i:i) = " "
      END DO

   END SUBROUTINE strip_control_codes

! **************************************************************************************************
!> \brief Copy the content of a \0-terminated C-string to a finite-length Fortran string
!>
!> The content of the new string may be truncated if the number of characters before the '\0'
!> in the source string exceed the length of the destination string. Based on the strlcpy C function.
!> \param fstring destination string
!> \param cstring source string
!> \return the number of characters copied
!> \author Tiziano Müller
! **************************************************************************************************
   FUNCTION strlcpy_c2f(fstring, cstring) RESULT(nchar)
      CHARACTER(LEN=*), INTENT(OUT)                      :: fstring
      CHARACTER(LEN=1, KIND=C_CHAR), INTENT(IN)          :: cstring(*)
      INTEGER                                            :: nchar

      DO nchar = 1, LEN(fstring)
         IF (cstring(nchar) == C_NULL_CHAR) EXIT
         fstring(nchar:nchar) = cstring(nchar)
      END DO

      nchar = nchar - 1
   END FUNCTION strlcpy_c2f

END MODULE string_utilities
