#include <time.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <glib.h>
#include <pthread.h>

#include "gm-debug.h"

typedef struct _LevelMap {
	gchar const *name;
	GmDebugLevel level;
} LevelMap;

static LevelMap level_mapping[] = {
	{"default", DEBUG_DEFAULT},
	{"mcp", DEBUG_MCP},
	{"all", DEBUG_ALL},
	{NULL, 0}
};

static gint debug_level = DEBUG_ALWAYS;
static pthread_mutex_t mutex_debug = PTHREAD_MUTEX_INITIALIZER;

static void
gm_debug_msg_real(FILE *f, struct tm *timet, gchar *line, va_list args) {
	fprintf(f, "[%02d:%02d:%02d] # ", timet->tm_hour, timet->tm_min,
			timet->tm_sec);
	vfprintf(f, line, args);
	fprintf(f, "\n");
}

void
gm_debug_msg(gint level, gchar *line, ...) {
	struct tm *timet;
	time_t timer;
	va_list args;
	
	pthread_mutex_lock(&mutex_debug);

	if ((debug_level | level) == debug_level) {
		va_start(args, line);
		timer = time(0);
		timet = localtime(&timer);

		if (level & DEBUG_ALWAYS) {
			gm_debug_msg_real(stdout, timet, line, args);
		}
		
		if ((level & DEBUG_ALWAYS) != level) {
			gm_debug_msg_real(stderr, timet, line, args);
		}

		va_end(args);
	}
	
	pthread_mutex_unlock(&mutex_debug);
}

void
gm_debug_set_level(gchar *level) {
	gchar **levels, **iter;
	debug_level = DEBUG_ALWAYS;
	LevelMap *map;
	
	if (level == NULL) {
		return;
	}
	
	levels = g_strsplit(level, ",", -1);
	
	pthread_mutex_lock(&mutex_debug);

	for (iter = levels; *iter; ++iter) {
		for (map = level_mapping; map->name; ++map) {
			if (strcasecmp(map->name, *iter) == 0) {
				debug_level = debug_level | map->level;
				break;
			}
		}
	}

	pthread_mutex_unlock(&mutex_debug);
	g_strfreev(levels);
}
