/*
 * Copyright (C) 2007-2010 Wayne Meissner
 *
 * This file is part of the JNR project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jnr.ffi.struct;

import jnr.ffi.Runtime;
import jnr.ffi.Struct;
import jnr.ffi.TstUtil;
import jnr.ffi.annotations.In;
import jnr.ffi.annotations.Out;
import jnr.ffi.annotations.Pinned;
import jnr.ffi.annotations.Transient;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;

public class AsciiStringFieldTest {
    public AsciiStringFieldTest() {
    }
    public class StringFieldStruct extends Struct {
        public final String string = new AsciiString(32);

        public StringFieldStruct() {
            super(runtime);
        }

        public StringFieldStruct(Runtime runtime) {
            super(runtime);
        }

    }
    public static interface TestLib {
        // This makes use of the string being the first field in the struct
        int string_equals(@Pinned @In @Transient StringFieldStruct s1, String s2);
        int copyByteBuffer(@Pinned @Out StringFieldStruct dst, @In byte[] src, int len);
        int copyByteBuffer(@Pinned @Out byte[] dst, @Pinned @In @Transient StringFieldStruct src, int len);
        int copyByteBuffer(@Pinned @Out StringBuilder dst, @Pinned @In @Transient StringFieldStruct src, int len);
    }
    static TestLib testlib;
    static Runtime runtime;
    @BeforeAll
    public static void setUpClass() throws Exception {
        testlib = TstUtil.loadTestLib(TestLib.class);
        runtime = Runtime.getRuntime(testlib);
    }

    @AfterAll
    public static void tearDownClass() throws Exception {
    }

    @BeforeEach
    public void setUp() {
    }

    @AfterEach
    public void tearDown() {
    }

    @Test
    public void stringFieldFirstInStruct() {
        StringFieldStruct s = new StringFieldStruct();
        final String MAGIC = "test";
        s.string.set(MAGIC);
        StringBuilder tmp = new StringBuilder(s.string.length());
        testlib.copyByteBuffer(tmp, s, s.string.length());
        assertEquals(MAGIC, tmp.toString(), "String not put into struct correctly");
    }

    public static final class StructWithStringByRef extends Struct {
        private final AsciiStringRef stringValue = new AsciiStringRef();

        public StructWithStringByRef() {
            super(runtime);
        }
    }

    @Test public void testStringByRef() {
        final String testValue = "some test string";
        final StructWithStringByRef struct = new StructWithStringByRef();

        struct.stringValue.set(testValue);

        assertEquals(testValue, struct.stringValue.get());
    }

    public static final class StructWithConsecutiveStringsByRef extends Struct {
        private final AsciiStringRef firstValue = new AsciiStringRef();
        private final AsciiStringRef secondValue = new AsciiStringRef();

        public StructWithConsecutiveStringsByRef() {
            super(runtime);
        }
    }

    @Test public void testStructWithConsecutiveStringsByRef() {
        final String testValue = "some test string";
        final StructWithConsecutiveStringsByRef struct = new StructWithConsecutiveStringsByRef();

        struct.firstValue.set("");
        struct.secondValue.set(testValue);

        assertEquals("", struct.firstValue.get());
        assertEquals(testValue, struct.secondValue.get());
    }
}
