"""Creates a unit agent-like context for invoking hook commands.

"""

# XXX issues
# still write some tests here


import logging
import os
import pipes
import stat
import sys

from twisted.internet.defer import inlineCallbacks

from juju.hooks.invoker import Invoker
from juju.hooks.protocol import UnitSettingsFactory
from juju.state.hook import HookContext
from juju.unit.lifecycle import HOOK_SOCKET_FILE


class InvokerUnitAgent(object):
    """Creates a unit-agent like context for running commands as hooks:

      * Initializes a client state cache and domain socket
      * Dynamically creates a wrapper script for the hook command
      * Runs wrapper script
    """
    def __init__(self, client, charm_dir, log_file, error_file, options):
        self.client = client
        self.charm_dir = charm_dir
        self.log_file = log_file
        self.error_file = error_file
        self.debug_file = sys.stderr
        self.options = options
        os.environ["JUJU_UNIT_NAME"] = self.unit_name = options.unit_name
        self.socket_path = os.path.join(self.charm_dir, HOOK_SOCKET_FILE)
        os.makedirs(os.path.join(self.charm_dir, "charm"))

        self.context = None
        self.invoker = None
        self.server_listen()

    @inlineCallbacks
    def start(self):
        """Build an Invoker for the execution of a hook."""

        # Output from hook commands is relayed by logging. Capture it
        # with this logger which is passed into the invoker. The
        # "jitsu-run-as-hook-output" name of the logger is completely
        # arbitrary, with this provision: it's not prefixed with
        # "juju." because otherwise if the log level for "juju" is
        # changed to above INFO, this capture doesn't actually occur!
        # (Hope that's clear.)
        logger = logging.getLogger("jitsu-run-as-hook-output")
        logger.propagate = False
        output_handler = logging.StreamHandler(self.log_file)
        output_handler.setLevel(logging.INFO)
        logger.addHandler(output_handler)

        # Capture any errors separately
        error_handler = logging.StreamHandler(self.error_file)
        error_handler.setLevel(logging.ERROR)
        logger.addHandler(error_handler)

        # Also grab debug output from unit agent if loglevel is below INFO
        if self.options.loglevel < logging.INFO:
            debug_handler = logging.StreamHandler(self.debug_file)
            debug_handler.setLevel(logging.DEBUG)
            debug_handler.setFormatter(logging.Formatter("%(asctime)s %(name)s:%(levelname)s %(message)s"))
            logger.addHandler(debug_handler)

        self.context = HookContext(self.client, self.unit_name)
        self.invoker = Invoker(
            self.context, None, "client_id", self.socket_path,
            self.charm_dir, logger)

        # When working with relation hook commands, setting
        # JUJU_REMOTE_UNIT to the same value as JUJU_UNIT_NAME
        # provides a useful default. A relation must still be
        # specified, however.
        self.invoker.environment["JUJU_REMOTE_UNIT"] = self.unit_name
        yield self.invoker.start()

    def get_context(self, client_id):
        """Required by the factory, always return this context"""
        return self.context

    def lookup_invoker(self, client_id):
        """Required by the factory, always return this invoker"""
        return self.invoker

    def server_listen(self):
        """Start Unix domain socket server for the constructed hook"""
        from twisted.internet import reactor

        self.server_factory = UnitSettingsFactory(
            self.get_context, self.lookup_invoker,
            logging.getLogger("juju.jitsu.do.unit-agent"))
        self.server_socket = reactor.listenUNIX(
            self.socket_path, self.server_factory)

    def stop(self):
        """Stop the process invocation."""
        self.server_socket.stopListening()

    def run_hook(self, command, args):
        """Given `commmand` and any `args`, builds a script that can be invoked.

        Both `command` and `args` are protected by shell quoting, as
        necessary.

        Returns a `Deferred` that can be yield upon to get the result,
        an exit code. Note that this exit code is always 0 for hook
        commands that complete, even if there's an error in their
        execution.
        """
        hook_path = os.path.join(self.charm_dir, "jitsu-run-as-hook-script")
        with open(hook_path, "w") as f:
            f.writelines([
                    "#!/bin/sh\n",
                    pipes.quote(command), " ",
                    " ".join(pipes.quote(arg) for arg in args), "\n"])
        os.chmod(hook_path, stat.S_IEXEC | stat.S_IREAD)
        return self.invoker(hook_path)
