"""Testing for the OpenStack provider interface"""

from juju.providers.openstack.tests import test_provider


class ProviderTestMixin(test_provider.ProviderTestMixin):

    from juju.providers.ec2.files import FileStorage as FileStorageClass
    from juju.providers.openstack_s3 import MachineProvider as ProviderClass

    test_environ = {
        "NOVA_URL": "https://environ.invalid",
        "NOVA_API_KEY": "env-key",
        "EC2_ACCESS_KEY": "env-key:env-project",
        "EC2_SECRET_KEY": "env-xxxx",
        "NOVA_PROJECT_ID": "env-project",
        "S3_URL": "https://environ.invalid:3333",
        }

    def get_config(self):
        config = super(ProviderTestMixin, self).get_config()
        config.update({
            "type": "openstack_s3",
            "combined-key": "key:project",
            "s3-uri": "https://testing.invalid:3333",
            })
        return config

    def get_client(self, provider):
        return provider.nova._client


class ProviderTests(test_provider.ProviderTests, ProviderTestMixin):

    def test_s3_params(self):
        """Config details get passed through to txaws S3 client correctly"""
        s3 = self.get_provider().s3
        self.assertEquals("https://testing.invalid:3333/",
            s3.endpoint.get_uri())
        self.assertEquals("key:project", s3.creds.access_key)
        self.assertEquals("xxxxxxxx", s3.creds.secret_key)

    def test_provider_attributes(self):
        """
        The provider environment name and config should be available as
        parameters in the provider.
        """
        provider = self.get_provider()
        self.assertEqual(provider.environment_name, self.environment_name)
        self.assertEqual(provider.config.get("type"), "openstack_s3")
        self.assertEqual(provider.provider_type, "openstack_s3")

    def test_config_environment_extraction(self):
        """
        The provider serialization loads keys as needed from the environment.

        Variables from the configuration take precendence over those from
        the environment, when serializing.
        """
        self.change_environment(**self.test_environ)
        provider = self.get_provider({
            "auth-mode": "keypair",
            "project-name": "other-project",
            "authorized-keys": "key-data",
            })
        serialized = provider.get_serialization_data()
        expected = {
            "auth-mode": "keypair",
            "access-key": "env-key",
            "secret-key": "env-xxxx",
            "auth-url": "https://environ.invalid",
            "project-name": "other-project",
            "authorized-keys": "key-data",
            "combined-key": "env-key:env-project",
            "s3-uri": "https://environ.invalid:3333",
            }
        self.assertEqual(expected, serialized)


class CheckCertsTests(test_provider.CheckCertsTests, ProviderTestMixin):

    def test_http_s3_url(self):
        provider, log = self.run_test({
            "s3-uri": "http://testing.invalid:3333",
            "ssl-hostname-verification": True,
            })
        self.assertTrue(provider.s3.endpoint.ssl_hostname_verification)
        self.assertEquals(True, self.get_client(provider).check_certs)
        self.assertIn("S3 API calls not using secure", log.getvalue())
