#! /usr/bin/python3
import os
import sys
import generator_utils as gen

template = """// @{generatedby}@
/* ///////////////////////// The MPI Bugs Initiative ////////////////////////

  Origin: MBI

  Description: @{shortdesc}@
    @{longdesc}@

  Version of MPI: Conforms to MPI 1.1, does not require MPI 2 implementation

BEGIN_MPI_FEATURES
  P2P!basic: @{p2pfeature}@
  P2P!nonblocking: @{ip2pfeature}@
  P2P!persistent: Lacking
  COLL!basic: @{collfeature}@
  COLL!nonblocking: @{icollfeature}@
  COLL!persistent: Lacking
  COLL!tools: Lacking
  RMA: Lacking
END_MPI_FEATURES

BEGIN_MBI_TESTS
  $ mpirun -np 2 ${EXE} 1
  | @{outcome1}@
  | @{errormsg1}@
  $ mpirun -np 2 ${EXE} 2
  | @{outcome2}@
  | @{errormsg2}@
END_MBI_TESTS
//////////////////////       End of MBI headers        /////////////////// */

#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>

#define N 10

int main(int argc, char **argv) {
  int nprocs = -1;
  int rank = -1;
  MPI_Status sta;
  int i=0;
  int root = 0;
  int stag=0;
  int rtag=0;
  int buff_size = N;

  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD, &nprocs);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  printf("Hello from rank %d \\n", rank);

  if (nprocs < 2)
    printf("MBI ERROR: This test needs at least 2 processes to produce a bug!\\n");

  if (argc < 2)
    printf("MBI ERROR: This test needs at least 1 argument to produce a bug!\\n");

  int dbs = sizeof(int)*nprocs; /* Size of the dynamic buffers for alltoall and friends */
  MPI_Comm newcom = MPI_COMM_WORLD;
  MPI_Datatype type = MPI_INT;
  MPI_Op op = MPI_SUM;

  int n = atoi(argv[1]);
  int buffer[N] = {42};

  @{init1}@
  @{init2}@

  if (rank == 0) {
    if ((n % 2) == 0) { @{errorcond}@
      @{operation1b}@
      @{fini1b}@
    } else {
      @{operation1a}@
      @{fini1a}@
    }
  } else @{addcond}@ {
    @{operation2}@
    @{fini2}@
  }

  @{free1}@
  @{free2}@

  MPI_Finalize();

  printf("Rank %d finished normally\\n", rank);
  return 0;
}
"""

# P2P
for s in gen.send + gen.isend:
    for r in gen.recv + gen.irecv:
        patterns = {}
        patterns = {'s': s, 'r': r}
        patterns['generatedby'] = f'DO NOT EDIT: this file was generated by {os.path.basename(sys.argv[0])}. DO NOT EDIT.'
        patterns['p2pfeature'] = 'Yes' if s in gen.send or r in gen.recv else 'Lacking'
        patterns['ip2pfeature'] = 'Yes' if s in gen.isend or r in gen.irecv else 'Lacking'
        patterns['collfeature'] = 'Lacking'
        patterns['icollfeature'] = 'Lacking'
        patterns['s'] = s
        patterns['r'] = r

        patterns['init1'] = gen.init[s]("1")
        patterns['operation1a'] = gen.operation[s]("1").replace("buf1", "buffer").replace("dest", "1")
        patterns['operation1b'] = gen.operation[s]("1").replace("buf1", "buffer").replace("dest", "1")
        patterns['fini1a'] = gen.fini[s]("1")
        patterns['fini1b'] = gen.fini[s]("1")
        patterns['free1'] = gen.free[s]("1")

        patterns['init2'] = gen.init[r]("2")
        patterns['operation2'] = gen.operation[r]("2").replace("buf2", "buffer").replace("src", "0")
        patterns['fini2'] = gen.fini[r]("2")
        patterns['free2'] = gen.free[r]("2")

        patterns['errorcond'] = ''
        patterns['addcond'] = 'if (rank == 1)'

        # Generate a correct matching
        replace = patterns.copy()
        replace['shortdesc'] = 'Correct call ordering.'
        replace['longdesc'] = 'Correct call ordering.'
        replace['outcome1'] = 'OK'
        replace['errormsg1'] = 'OK'
        replace['outcome2'] = 'OK'
        replace['errormsg2'] = 'OK'
        gen.make_file(template, f'InputHazardCallOrdering_{r}_{s}_ok.c', replace)

        # Generate the incorrect matching
        replace = patterns.copy()
        replace['shortdesc'] = 'Missing Send function.'
        replace['longdesc'] = 'Missing Send function call for a path depending to input, a deadlock is created.'
        replace['outcome1'] = 'OK'
        replace['errormsg1'] = 'OK'
        replace['outcome2'] = 'ERROR: IHCallMatching'
        replace['errormsg2'] = 'P2P mistmatch. Missing @{r}@ at @{filename}@:@{line:MBIERROR}@.'
        replace['errorcond'] = '/* MBIERROR */'
        replace['operation1b'] = ''
        replace['fini1b'] = ''
        gen.make_file(template, f'InputHazardCallOrdering_{r}_{s}_nok.c', replace)

# COLLECTIVE
for c in gen.coll:
    patterns = {}
    patterns = {'c': c}
    patterns['generatedby'] = f'DO NOT EDIT: this file was generated by {os.path.basename(sys.argv[0])}. DO NOT EDIT.'
    patterns['p2pfeature'] = 'Lacking'
    patterns['ip2pfeature'] = 'Lacking'
    patterns['collfeature'] = 'Yes' if c in gen.coll else 'Lacking'
    patterns['icollfeature'] = 'Yes' if c in gen.icoll else 'Lacking'
    patterns['c'] = c

    patterns['init1'] = gen.init[c]("1")
    patterns['operation1a'] = gen.operation[c]("1")
    patterns['operation1b'] = gen.operation[c]("1")
    patterns['fini1a'] = gen.fini[c]("1")
    patterns['fini1b'] = gen.fini[c]("1")
    patterns['free1'] = gen.free[c]("1")

    patterns['init2'] = gen.init[c]("2")
    patterns['operation2'] = gen.operation[c]("2")
    patterns['fini2'] = gen.fini[c]("2")
    patterns['free2'] = gen.free[c]("2")

    patterns['errorcond'] = ''
    patterns['addcond'] = ''

    # Generate a correct matching
    replace = patterns.copy()
    replace['shortdesc'] = 'Correct call ordering.'
    replace['longdesc'] = 'Correct call ordering.'
    replace['outcome1'] = 'OK'
    replace['errormsg1'] = 'OK'
    replace['outcome2'] = 'OK'
    replace['errormsg2'] = 'OK'
    gen.make_file(template, f'InputHazardCallOrdering_{c}_ok.c', replace)

    # Generate the incorrect matching
    replace = patterns.copy()
    replace['shortdesc'] = 'Missing collective function call.'
    replace['longdesc'] = 'Missing collective function call for a path depending to input, a deadlock is created.'
    replace['outcome1'] = 'OK'
    replace['errormsg1'] = 'OK'
    replace['outcome2'] = 'ERROR: IHCallMatching'
    replace['errormsg2'] = 'P2P mistmatch. Missing @{c}@ at @{filename}@:@{line:MBIERROR}@.'
    replace['errorcond'] = '/* MBIERROR */'
    replace['operation1b'] = ''
    replace['fini1b'] = ''
    gen.make_file(template, f'InputHazardCallOrdering_{c}_nok.c', replace)
