/*
 * synergy -- mouse and keyboard sharing utility
 * Copyright (C) 2012 Synergy Si Ltd.
 * Copyright (C) 2012 Nick Bolton
 * 
 * This package is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * found in the file COPYING that should have accompanied this file.
 * 
 * This package is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ipc/IpcLogOutputter.h"

#include "ipc/IpcServer.h"
#include "ipc/IpcMessage.h"
#include "ipc/Ipc.h"
#include "ipc/IpcClientProxy.h"
#include "mt/Thread.h"
#include "arch/Arch.h"
#include "arch/XArch.h"
#include "base/Event.h"
#include "base/EventQueue.h"
#include "base/TMethodEventJob.h"
#include "base/TMethodJob.h"

// limit number of log lines sent in one message.
#define MAX_SEND 100

IpcLogOutputter::IpcLogOutputter(IpcServer& ipcServer) :
m_ipcServer(ipcServer),
m_bufferMutex(ARCH->newMutex()),
m_sending(false),
m_running(true),
m_notifyCond(ARCH->newCondVar()),
m_notifyMutex(ARCH->newMutex()),
m_bufferWaiting(false)
{
	m_bufferThread = new Thread(new TMethodJob<IpcLogOutputter>(
		this, &IpcLogOutputter::bufferThread));
}

IpcLogOutputter::~IpcLogOutputter()
{
	m_running = false;
	notifyBuffer();
	m_bufferThread->wait(5);

	ARCH->closeMutex(m_bufferMutex);
	delete m_bufferThread;

	ARCH->closeCondVar(m_notifyCond);
	ARCH->closeMutex(m_notifyMutex);
}

void
IpcLogOutputter::open(const char* title)
{
}

void
IpcLogOutputter::close()
{
}

void
IpcLogOutputter::show(bool showIfEmpty)
{
}

bool
IpcLogOutputter::write(ELevel level, const char* text)
{
	return write(level, text, false);
}

bool
IpcLogOutputter::write(ELevel, const char* text, bool force)
{
	// TODO: discard based on thread id? hmm...
	// sending the buffer generates log messages, which we must throw
	// away (otherwise this would cause recursion). this is just a drawback
	// of logging this way. there is also the risk that this could throw
	// away log messages not generated by the ipc, but it seems like it
	// would be difficult to distinguish (other than looking at the stack
	// trace somehow). perhaps a file stream might be a better option :-/
	if (m_sending && !force) {

		// ignore events from the buffer thread (would cause recursion).
		if (Thread::getCurrentThread().getID() == m_bufferThreadId) {
			return true;
		}
	}

	appendBuffer(text);
	notifyBuffer();
	return true;
}

void
IpcLogOutputter::appendBuffer(const String& text)
{
	ArchMutexLock lock(m_bufferMutex);
	m_buffer.push(text);
}

void
IpcLogOutputter::bufferThread(void*)
{
	ArchMutexLock lock(m_notifyMutex);
	m_bufferThreadId = m_bufferThread->getID();

	try {
		while (m_running) {

			if (m_ipcServer.hasClients(kIpcClientGui)) {

				// buffer is sent in chunks, so keep sending until it's
				// empty (or the program has stopped in the meantime).
				while (m_running && !m_buffer.empty()) {
					sendBuffer();
				}
			}

			// program may be stopping while we were in the send loop.
			if (!m_running) {
				break;
			}

			m_bufferWaiting = true;
			ARCH->waitCondVar(m_notifyCond, m_notifyMutex, -1);
			m_bufferWaiting = false;
		}
	}
	catch (XArch& e) {
		LOG((CLOG_ERR "ipc log buffer thread error, %s", e.what()));
	}

	LOG((CLOG_DEBUG "ipc log buffer thread finished"));
}

void
IpcLogOutputter::notifyBuffer()
{
	if (!m_bufferWaiting) {
		return;
	}
	ArchMutexLock lock(m_notifyMutex);
	ARCH->broadcastCondVar(m_notifyCond);
}

String
IpcLogOutputter::getChunk(size_t count)
{
	ArchMutexLock lock(m_bufferMutex);

	if (m_buffer.size() < count) {
		count = m_buffer.size();
	}

	String chunk;
	for (size_t i = 0; i < count; i++) {
		chunk.append(m_buffer.front());
		chunk.append("\n");
		m_buffer.pop();
	}
	return chunk;
}

void
IpcLogOutputter::sendBuffer()
{
	IpcLogLineMessage message(getChunk(MAX_SEND));

	m_sending = true;
	m_ipcServer.send(message, kIpcClientGui);
	m_sending = false;
}
