# - Try to find Poppler and its components
#
# Valid components are: cpp glib qt5 qt6
#
# Once done this will define
#
#  Poppler_FOUND - system has Poppler
#  Poppler_NEEDS_FONTCONFIG - A boolean indicating if libpoppler depends on libfontconfig
#  Poppler_PRIVATE_INCLUDE_DIRS - the include directories for Poppler private headers (if they exist)
#  Poppler_LIBRARIES - Link these to use Poppler
#  Poppler_<C>_FOUND - system has Poppler component <C>
#  Poppler_<C>_INCLUDE_DIRS - the include directories for component <C> headers
#  Poppler_<C>_LIBRARIES  - Link this to use the component <C>
#
# In addition, the following IMPORT targets will be created
#
#  Poppler::poppler
#  Poppler::poppler-<C> for each component <C>
#
# Redistribution and use of this file is allowed according to the terms of the
# MIT license. For details see the file COPYING-CMAKE-MODULES.

include(FindPackageHandleStandardArgs)
include(FindPackageMessage)

# -----------------
# Find Poppler core
# -----------------

# use pkg-config to get the directories and then use these values in the
# FIND_PATH() and FIND_LIBRARY() calls
if (NOT WIN32)
  find_package(PkgConfig QUIET)
  pkg_check_modules(Poppler_PKG QUIET poppler)
endif (NOT WIN32)

# Find libpoppler
find_library(Poppler_LIBRARY NAMES poppler ${Poppler_PKG_LIBRARIES}
  PATHS
    /usr/local
    /usr
  HINTS
    ${Poppler_PKG_LIBRARY_DIRS} # Generated by pkg-config
  PATH_SUFFIXES
    lib64
    lib
)

# Find include directory for private headers (optional)
find_path(Poppler_PRIVATE_INCLUDE_DIR NAMES poppler-config.h
  PATHS
    /usr/local/include
    /usr/include
  HINTS
    ${Poppler_PKG_INCLUDE_DIRS} # Generated by pkg-config
  PATH_SUFFIXES
    poppler
)

# If private headers were found, determine poppler's version
if (Poppler_PRIVATE_INCLUDE_DIR)
  find_package_message(Poppler_PRIVATE_HEADERS "Found Poppler private headers: ${Poppler_PRIVATE_INCLUDE_DIR}" "${Poppler_PRIVATE_INCLUDE_DIR}")

  file(STRINGS "${Poppler_PRIVATE_INCLUDE_DIR}/poppler-config.h" Poppler_CONFIG_H REGEX "^#define POPPLER_VERSION \"[0-9.]+\"$")

  if(Poppler_CONFIG_H)
    string(REGEX REPLACE "^.*POPPLER_VERSION \"([0-9.]+)\"$" "\\1" Poppler_VERSION_STRING "${Poppler_CONFIG_H}")
    string(REGEX REPLACE "^([0-9]+).*$" "\\1" Poppler_VERSION_MAJOR "${Poppler_VERSION_STRING}")
    string(REGEX REPLACE "^${Poppler_VERSION_MAJOR}\\.([0-9]+).*$" "\\1" Poppler_VERSION_MINOR  "${Poppler_VERSION_STRING}")
    string(REGEX REPLACE "^${Poppler_VERSION_MAJOR}\\.${Poppler_VERSION_MINOR}\\.([0-9]+)$" "\\1" Poppler_VERSION_PATCH "${Poppler_VERSION_STRING}")

    if (Poppler_VERSION_MINOR STREQUAL Poppler_VERSION_STRING)
      unset(Poppler_VERSION_MINOR)
    endif()
    if (Poppler_VERSION_PATCH STREQUAL Poppler_VERSION_STRING)
      unset(Poppler_VERSION_PATCH)
    endif()
  endif()
endif ()


# Scan poppler libraries for dependencies on Fontconfig
include(GetPrerequisites)
mark_as_advanced(gp_cmd)
get_prerequisites("${Poppler_LIBRARY}" Poppler_PREREQS TRUE FALSE "" "")
if ("${Poppler_PREREQS}" MATCHES "fontconfig")
  set(Poppler_NEEDS_FONTCONFIG TRUE)
else ()
  set(Poppler_NEEDS_FONTCONFIG FALSE)
endif ()

# -----------------------
# Find Poppler components
# -----------------------

foreach (cmp IN LISTS Poppler_FIND_COMPONENTS)
  set(label "Poppler_${cmp}")
  set(pkg "poppler-${cmp}")

  if (NOT WIN32)
    pkg_check_modules(${label}_PKG QUIET ${pkg})
  endif (NOT WIN32)

  # Find library
  find_library(${label}_LIBRARY NAMES ${pkg} ${${label}_PKG_LIBRARIES}
    PATHS
      /usr/local
      /usr
    HINTS
      ${${label}_PKG_LIBRARY_DIRS} # Generated by pkg-config
    PATH_SUFFIXES
      lib64
      lib
  )

  # Find include directory
  if ("${cmp}" STREQUAL qt6)
    set(${pkg}_header poppler-qt6.h)
    # NB: find_package(Qt6) changes pkg in our scope, so back it up and restore
    # it afterwards
    set(_pkg "${pkg}")
    find_package(Qt6 REQUIRED COMPONENTS Core Gui Xml)
    set(pkg "${_pkg}")
  elseif ("${cmp}" STREQUAL qt5)
    set(${pkg}_header poppler-qt5.h)
    find_package(Qt5 REQUIRED COMPONENTS Core Gui Xml)
  else ()
    set(${pkg}_header poppler-document.h)
  endif ()

  find_path(${label}_INCLUDE_DIR NAMES ${${pkg}_header}
    PATHS
      /usr/local/include
      /usr/include
    HINTS
      ${${label}_PKG_INCLUDE_DIRS} # Generated by pkg-config
    PATH_SUFFIXES
      poppler/${cmp}
  )

  # Find version
  if (EXISTS "${${label}_INCLUDE_DIR}/poppler-version.h")
    file(STRINGS "${${label}_INCLUDE_DIR}/poppler-version.h" ${label}_CONFIG_H REGEX "^#define POPPLER_VERSION \"[0-9.]+\"$")

    if(${label}_CONFIG_H)
      string(REGEX REPLACE "^.*POPPLER_VERSION \"([0-9.]+)\"$" "\\1" ${label}_VERSION_STRING "${${label}_CONFIG_H}")
      string(REGEX REPLACE "^([0-9]+).*$" "\\1" ${label}_VERSION_MAJOR "${${label}_VERSION_STRING}")
      string(REGEX REPLACE "^${${label}_VERSION_MAJOR}\\.([0-9]+).*$" "\\1" ${label}_VERSION_MINOR  "${${label}_VERSION_STRING}")
      string(REGEX REPLACE "^${${label}_VERSION_MAJOR}\\.${${label}_VERSION_MINOR}\\.([0-9]+)$" "\\1" ${label}_VERSION_PATCH "${${label}_VERSION_STRING}")

      if (${label}_VERSION_MINOR STREQUAL ${label}_VERSION_STRING)
        unset(${label}_VERSION_MINOR)
      endif()
      if (${label}_VERSION_PATCH STREQUAL ${label}_VERSION_STRING)
        unset(${label}_VERSION_PATCH)
      endif()
    endif()
  endif ()

  if (CMAKE_VERSION VERSION_LESS "3.17")
    find_package_handle_standard_args(${label} FOUND_VAR ${label}_FOUND REQUIRED_VARS ${label}_LIBRARY ${label}_INCLUDE_DIR VERSION_VAR ${label}_VERSION_STRING)
  else ()
    find_package_handle_standard_args(${label} FOUND_VAR ${label}_FOUND REQUIRED_VARS ${label}_LIBRARY ${label}_INCLUDE_DIR VERSION_VAR ${label}_VERSION_STRING NAME_MISMATCHED)
  endif ()

  if (${label}_FOUND)
    set(${label}_INCLUDE_DIRS "${${label}_INCLUDE_DIR}" "${Poppler_INCLUDE_DIR}")
    set(${label}_LIBRARIES "${${label}_LIBRARY}" "${Poppler_LIBRARY}")

    if (NOT TARGET Poppler::poppler-${cmp})
      add_library(Poppler::poppler-${cmp} UNKNOWN IMPORTED)
      set_target_properties(Poppler::poppler-${cmp} PROPERTIES
        IMPORTED_LOCATION "${${label}_LIBRARY}"
        INTERFACE_INCLUDE_DIRECTORIES "${${label}_INCLUDE_DIR}"
      )
      if ("${cmp}" STREQUAL "qt5")
        # NB: It is important to list Poppler::poppler last as it may share
        # dependencies with Qt (when compiled statically); particularly harfbuzz
        # and freetype don't play nicely if poppler is not listed last
        set_target_properties(Poppler::poppler-${cmp} PROPERTIES INTERFACE_LINK_LIBRARIES "Qt5::Core;Qt5::Gui;Qt5::Xml;Poppler::poppler")
      else ()
        set_target_properties(Poppler::poppler-${cmp} PROPERTIES INTERFACE_LINK_LIBRARIES Poppler::poppler)
      endif ()
    endif ()
  endif ()
endforeach ()

# ---------
# Finish up
# ---------

find_package_handle_standard_args(Poppler FOUND_VAR Poppler_FOUND REQUIRED_VARS Poppler_LIBRARY VERSION_VAR Poppler_VERSION_STRING HANDLE_COMPONENTS)

if (Poppler_FOUND)
  set(Poppler_INCLUDE_DIRS "${Poppler_INCLUDE_DIR}")
  set(Poppler_LIBRARIES "${Poppler_LIBRARY}")
  if (NOT TARGET Poppler::poppler)
    add_library(Poppler::poppler UNKNOWN IMPORTED)
    set_target_properties(Poppler::poppler PROPERTIES IMPORTED_LOCATION "${Poppler_LIBRARIES}")
    if (Poppler_PRIVATE_INCLUDE_DIR)
      set_target_properties(Poppler::poppler PROPERTIES
        INTERFACE_INCLUDE_DIRECTORIES "${Poppler_PRIVATE_INCLUDE_DIR}"
        HAS_PRIVATE_HEADERS On
      )
    endif ()

    set(_poppler_additional_libs "")
    set(Poppler_ADDITIONAL_DEPENDENCIES "" CACHE STRING "Additional libraries not found by CMake")
    MARK_AS_ADVANCED(Poppler_ADDITIONAL_DEPENDENCIES)

    if (Poppler_NEEDS_FONTCONFIG)
      find_package(Fontconfig)
      list(APPEND _poppler_additional_libs "${FONTCONFIG_LIBRARIES}")
    endif ()
    if (Poppler_ADDITIONAL_DEPENDENCIES)
      list(APPEND _poppler_additional_libs "${Poppler_ADDITIONAL_DEPENDENCIES}")
    endif ()
    set_target_properties(Poppler::poppler PROPERTIES INTERFACE_LINK_LIBRARIES "${_poppler_additional_libs}")
  endif ()
endif ()
