/*
 * MIT Kerberos interface for WebAuth.
 *
 * This file is *included* (via the preprocessor) in krb5.c for systems that
 * use MIT Kerberos.  If you make any changes here, you probably also need to
 * make a corresponding change to krb5-heimdal.c for systems with Heimdal.
 *
 * Written by Russ Allbery <rra@stanford.edu>
 * Based on the original Kerberos support code by Roland Schemers
 * Copyright 2002, 2003, 2006, 2009, 2010, 2012, 2013
 *     The Board of Trustees of the Leland Stanford Junior University
 *
 * See LICENSE for licensing terms.
 */


/*
 * Reverse the order of the Kerberos credential flag bits.  This converts from
 * the Heimdal memory format to the correct format to adjust for incorrect
 * credentials generated by versions of WebAuth prior to 4.4.0 built against
 * Heimdal.
 */
static int32_t
swap_flag_bits(int32_t flags)
{
    int32_t result = 0;
    unsigned int i;

    for (i = 0; i < 32; i++) {
        result = (result << 1) | (flags & 1);
        flags = flags >> 1;
    }
    return result;
}


/*
 * Take a single Kerberos credential and serialize it into a buffer, using the
 * encoding required for putting it into tokens.  output will be a pointer to
 * newly allocated memory, and length will be set to the encoded length.
 * expiration will be set to the expiration time of the ticket.  Returns a
 * WA_ERR code.
 */
static int
encode_creds(struct webauth_context *ctx, struct webauth_krb5 *kc,
             krb5_creds *creds, void **output, size_t *length,
             time_t *expiration)
{
    int s;
    struct wai_krb5_cred data;

    /* Start by copying the credential data into our standard struct. */
    memset(&data, 0, sizeof(data));
    s = encode_principal(ctx, kc, creds->client, &data.client_principal);
    if (s != WA_ERR_NONE)
        return s;
    s = encode_principal(ctx, kc, creds->server, &data.server_principal);
    if (s != WA_ERR_NONE)
        return s;
    data.keyblock_enctype  = creds->keyblock.enctype;
    data.keyblock_data     = creds->keyblock.contents;
    data.keyblock_data_len = creds->keyblock.length;
    data.auth_time         = creds->times.authtime;
    data.start_time        = creds->times.starttime;
    data.end_time          = creds->times.endtime;
    if (expiration != NULL)
        *expiration = creds->times.endtime;
    data.renew_until       = creds->times.renew_till;
    data.is_skey           = creds->is_skey;
    data.flags             = creds->ticket_flags;
    if (creds->addresses != NULL && creds->addresses[0] != NULL) {
        size_t n, i, size;

        for (n = 0; creds->addresses[n] != NULL; n++)
            ;
        data.address_count = n;
        size = n * sizeof(struct wai_krb5_cred_address);
        data.address = apr_palloc(kc->pool, size);
        for (i = 0; i < n; i++) {
            data.address[i].type = creds->addresses[i]->addrtype;
            data.address[i].data = creds->addresses[i]->contents;
            data.address[i].data_len = creds->addresses[i]->length;
        }
    }
    if (creds->ticket.length > 0) {
        data.ticket     = creds->ticket.data;
        data.ticket_len = creds->ticket.length;
    }
    if (creds->second_ticket.length > 0) {
        data.second_ticket     = creds->second_ticket.data;
        data.second_ticket_len = creds->second_ticket.length;
    }
    if (creds->authdata != NULL && creds->authdata[0] != NULL) {
        size_t n, i, size;

        for (n = 0; creds->authdata[n] != NULL; n++)
            ;
        data.authdata_count = n;
        size = n * sizeof(struct wai_krb5_cred_authdata);
        data.authdata = apr_palloc(kc->pool, size);
        for (i = 0; i < n; i++) {
            data.authdata[i].type = creds->authdata[i]->ad_type;
            data.authdata[i].data = creds->authdata[i]->contents;
            data.authdata[i].data_len = creds->authdata[i]->length;
        }
    }

    /* All done.  Do the attribute encoding. */
    return wai_encode(ctx, wai_krb5_cred_encoding, &data, output, length);
}


/*
 * Take a serialized Kerberos credential and decode it into a krb5_creds
 * structure.  creds will point to newly-allocated pool memory.
 *
 * Be very cautious of memory management here.  Nearly all of the credential
 * structure will be allocated from pool memory, and therefore the credential
 * must not be freed with the normal Kerberos memory calls.  However, the
 * client and server principals will be allocated by the Kerberos library and
 * will need to be freed.
 */
static int
decode_creds(struct webauth_context *ctx, struct webauth_krb5 *kc,
             const void *input, size_t length, krb5_creds *creds)
{
    struct wai_krb5_cred data;
    int s;
    size_t size, i;

    /*
     * Decode the input into the credential struct and then copy it into
     * the data structure used by the library.
     */
    memset(&data, 0, sizeof(data));
    s = wai_decode(ctx, wai_krb5_cred_encoding, input, length, &data);
    if (s != WA_ERR_NONE)
        return s;
    memset(creds, 0, sizeof(krb5_creds));
    if (data.client_principal != NULL) {
        s = decode_principal(ctx, kc, data.client_principal, &creds->client);
        if (s != WA_ERR_NONE)
            return s;
    }
    if (data.client_principal != NULL) {
        s = decode_principal(ctx, kc, data.server_principal, &creds->server);
        if (s != WA_ERR_NONE)
            return s;
    }
    creds->keyblock.magic = KV5M_KEYBLOCK;
    creds->keyblock.enctype = data.keyblock_enctype;
    creds->keyblock.contents = data.keyblock_data;
    creds->keyblock.length = data.keyblock_data_len;
    creds->times.authtime = data.auth_time;
    creds->times.starttime = data.start_time;
    creds->times.endtime = data.end_time;
    creds->times.renew_till = data.renew_until;
    creds->is_skey = data.is_skey;
    if (data.address_count > 0) {
        size = (data.address_count + 1) * sizeof(krb5_address *);
        creds->addresses = apr_pcalloc(kc->pool, size);
        for (i = 0; i < data.address_count; i++) {
            creds->addresses[i] = apr_pcalloc(kc->pool, sizeof(krb5_address));
            creds->addresses[i]->magic = KV5M_ADDRESS;
            creds->addresses[i]->addrtype = data.address[i].type;
            creds->addresses[i]->contents = data.address[i].data;
            creds->addresses[i]->length = data.address[i].data_len;
        }
        creds->addresses[i] = NULL;
    }
    if (data.ticket != NULL) {
        creds->ticket.magic = KV5M_DATA;
        creds->ticket.data = data.ticket;
        creds->ticket.length = data.ticket_len;
    }
    if (data.second_ticket != NULL) {
        creds->second_ticket.magic = KV5M_DATA;
        creds->second_ticket.data = data.second_ticket;
        creds->second_ticket.length = data.second_ticket_len;
    }
    if (data.authdata_count > 0) {
        size = (data.authdata_count + 1) * sizeof(krb5_authdata *);
        creds->authdata = apr_pcalloc(kc->pool, size);
        for (i = 0; i < data.authdata_count; i++) {
            creds->authdata[i] = apr_palloc(kc->pool, sizeof(krb5_authdata));
            creds->authdata[i]->magic = KV5M_AUTHDATA;
            creds->authdata[i]->ad_type = data.authdata[i].type;
            creds->authdata[i]->contents = data.authdata[i].data;
            creds->authdata[i]->length = data.authdata[i].data_len;
        }
        creds->authdata[i] = NULL;
    }

    /*
     * The portable representation of the flag bits has forwardable near the
     * most significant end, which matches how MIT stores it internally.
     * However, unfortunately, we used to store the flag bits on the wire in
     * memory format, and Heimdal uses the oppposite memory format.
     * Therefore, credentials written by old versions of WebAuth built with
     * Heimdal will have the flag bits reversed.
     *
     * Try to figure out if we did that by seeing if any of the high flag bits
     * are set and, if not, swap the bits for backwards compatibility with
     * older versions of WebAuth built against Heimdal.  This relies on the
     * fact that credentials always have at least one flag set, and all the
     * currently used flags are in the top half of the portable
     * representation.
     *
     * WebAuth 4.4.0 and later will always write out the flag bits in the
     * correct order.  This code could theoretically be simplified to never
     * swap if nothing older is still in the wild.
     */
    if (data.flags & 0xffff0000)
        creds->ticket_flags = data.flags;
    else
        creds->ticket_flags = swap_flag_bits(data.flags);

    return WA_ERR_NONE;
}
