/* nm_filetype_thread.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2011 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "nm_filetype_thread.hh"
#include "simplelist.hh"
#include "fileentry.hh"
#include <aguix/lowlevelfunc.h>
#include "datei.h"
#include "fileentry_typecheck.hh"
#include "wcfiletype.hh"
#include "worker.h"

NM_Filetype_Thread::NM_Filetype_Thread() : m_worker( NULL ), m_running( 0 ),
                                           m_order( THREAD_STOP ), m_status( THREAD_STOP )
{
    m_filetype_copy.filetypes = new List();
}

NM_Filetype_Thread::~NM_Filetype_Thread()
{
    res_list_clear();

    clear_filetypes();
    delete m_filetype_copy.filetypes;
}

void NM_Filetype_Thread::setWorker( Worker *worker )
{
    m_worker = worker;
}

int NM_Filetype_Thread::slave_checkFiletype( const std::string &fullname, bool dontCheckContent, checkFiletypeRes_t &res_return )
{
    FileEntry *fe1;
    int pos = -1;
    WCFiletype *ft, *parentft;
    std::vector<unsigned int> *v = NULL;

    m_filetype_copy.filetype_ex.lock();

    fe1 = new FileEntry();
    fe1->fullname = dupstring( fullname.c_str() );
    fe1->name = Datei::getFilenameFromPath( fullname.c_str() );

    //TODO start check even for corrupt entries without info?
    if( fe1->readInfos() == 0 ) {
        ft = FileEntryTypeCheck::checkFiletype( *fe1, m_filetype_copy.filetypes, dontCheckContent, &condparser );
        if ( ft != NULL ) {
            // first get vector with filetype position for each child
            v = ft->getTypePos();
            // now find root filetype
      
            for ( parentft = ft; parentft->getParentType() != NULL; parentft = parentft->getParentType() );
            pos = m_filetype_copy.filetypes->getIndex( parentft );
            if ( pos >= 0 ) {
                if ( v != NULL ) v->push_back( (unsigned int)pos );
            } else {
                if ( v != NULL ) {
                    delete v;
                    v = NULL;
                }
            }
        }
    }
    res_return.v = v;

    if ( fe1->getCustomColorSet() != 0 ) {
        try {
            res_return.custom_color.reset( new FileEntryCustomColor( fe1->getCustomColor() ) );
        } catch ( int ) {
        }
    }
    delete fe1;

    m_filetype_copy.filetype_ex.unlock();
    return 0;
}

int NM_Filetype_Thread::switchOrder( thread_order_t neworder )
{
    if ( m_order != neworder ) {
        m_ordervar_cv.lock();
        m_order = neworder;
        m_ordervar_cv.signal();
        m_ordervar_cv.unlock();

        m_statusvar_cv.lock();
        while ( m_status != neworder ) {
            m_statusvar_cv.wait();
        }
        m_statusvar_cv.unlock();
    }
    return 0;
}

int NM_Filetype_Thread::switchStatus( thread_order_t newstatus )
{
    m_statusvar_cv.lock();
    m_status = newstatus;
    m_statusvar_cv.signal();
    m_statusvar_cv.unlock();
    return 0;
}

int NM_Filetype_Thread::run()
{
    ft_recres_list_t *te;
    bool ende;
    int erg;
    bool dontCheckContent = false;

    if ( m_running != 0 ) {
        fprintf( stderr, "Worker: another thread already running!\n");
        return 1;
    }
    m_running = 1;

#ifdef DEBUG
    printf("entering slave handler\n");
#endif

    for( ende = false; ende == false; ) {

        switch ( m_status ) {
            case NM_Filetype_Thread::THREAD_RUN:
#ifdef DEBUG
                printf("waiting for element\n");
#endif

                m_ordervar_cv.lock();
                // wait for next element or stop
                while ( m_entries_to_check.empty() && m_order == NM_Filetype_Thread::THREAD_RUN )
                    m_ordervar_cv.wait();

#ifdef DEBUG
                printf("wait finished\n");
#endif

                if ( m_order != NM_Filetype_Thread::THREAD_RUN ) {
                    // new command
                    // switch mode for new command
                    if ( m_order == NM_Filetype_Thread::THREAD_EXIT ) {
                        switchStatus( NM_Filetype_Thread::THREAD_EXIT );
                        ende = true;
                    } else if ( m_order == NM_Filetype_Thread::THREAD_STOP ) {
                        switchStatus( NM_Filetype_Thread::THREAD_STOP );
                    }
                    m_ordervar_cv.unlock();
                } else {
                    // an element to check
                    std::string filename;

                    if ( ! m_entries_to_check.empty() ) {
                        check_entry_t e1 = m_entries_to_check.front();
                        m_entries_to_check.pop_front();
                        filename = e1.name;
                        dontCheckContent = e1.dont_check_content;
                        erg = 0;
                    } else {
                        erg = 1;
                    }
                    m_ordervar_cv.unlock();

                    if ( erg == 0 && ! filename.empty() ) {
                        NM_Filetype_Thread::checkFiletypeRes_t res;
#ifdef DEBUG
                        //            printf("erkenne %s\n",filename);
#endif
                        slave_checkFiletype( filename, dontCheckContent, res );
                        recreslist.lock();
    
                        te = new ft_recres_list_t;
                        te->name = filename;
                        te->ft_index = res.v;
                        if ( res.custom_color.get() != NULL &&
                             res.custom_color->getColorSet() != 0 ) {
                            te->custom_color = res.custom_color;
                        }
    
                        recreslist.put_locked( te );
                        recreslist.unlock();

                        if ( m_worker ) m_worker->wakeupMainLoop();
                    }
                }
                break;
            case NM_Filetype_Thread::THREAD_STOP:
                m_ordervar_cv.lock();
                while ( m_order == NM_Filetype_Thread::THREAD_STOP ) {
                    m_ordervar_cv.wait();
                }
                m_ordervar_cv.unlock();
                switchStatus( NM_Filetype_Thread::THREAD_RUN );
                break;
            default:
                break;
        }
    }
    
#ifdef DEBUG
    printf("leaving slave handler\n");
#endif
    return 0;
}

NM_Filetype_Thread::ft_recres_list::ft_recres_list()
{
}

NM_Filetype_Thread::ft_recres_list::~ft_recres_list()
{
    ft_recres_list_t *rte;

    lock();
    while ( isEmpty_locked() == false ) {
        rte = remove_locked();
        if ( rte != NULL ) {
            delete rte;
        }
    }
    unlock();
}

bool NM_Filetype_Thread::ft_recres_list::isEmpty_locked()
{
    if ( entries.empty() ) return true;
    return false;
}

int NM_Filetype_Thread::ft_recres_list::put_locked( ft_recres_list_t *elem )
{
    if ( elem == NULL ) return -1;

    entries.push_back( elem );
    return 0;
}

NM_Filetype_Thread::ft_recres_list_t *NM_Filetype_Thread::ft_recres_list::remove_locked()
{
    ft_recres_list_t *te;

    if ( entries.empty() ) return NULL;

    te = entries.front();
    entries.pop_front();
    return te;
}

void NM_Filetype_Thread::ft_recres_list::lock()
{
    ex.lock();
}

void NM_Filetype_Thread::ft_recres_list::unlock()
{
    ex.unlock();
}

void NM_Filetype_Thread::res_list_clear()
{
    ft_recres_list_t *rte;

    recreslist.lock();
    while ( recreslist.isEmpty_locked() == false ) {
        // alle Resultate entnehmen
        rte = recreslist.remove_locked();
        if ( rte != NULL ) {
            delete rte;
        }
    }
    recreslist.unlock();
}

void NM_Filetype_Thread::res_list_lock()
{
    recreslist.lock();
}

void NM_Filetype_Thread::res_list_unlock()
{
    recreslist.unlock();
}

NM_Filetype_Thread::ft_recres_list_t *NM_Filetype_Thread::res_list_remove_locked()
{
    return recreslist.remove_locked();
}

bool NM_Filetype_Thread::res_list_is_empty_locked()
{
    return recreslist.isEmpty_locked();
}

void NM_Filetype_Thread::putRequest( const std::string &name,
                                     bool dont_check_content )
{
    m_ordervar_cv.lock();

    check_entry_t e1;
    e1.name = name;
    e1.dont_check_content = dont_check_content;
    m_entries_to_check.push_back( e1 );
    m_ordervar_cv.signal();
    m_ordervar_cv.unlock();
}

void NM_Filetype_Thread::clearRequests()
{
    m_ordervar_cv.lock();
    m_entries_to_check.clear();
    m_ordervar_cv.signal();
    m_ordervar_cv.unlock();
}

void NM_Filetype_Thread::copy_filetypes( List *filetypes )
{
    WCFiletype *ft;
    int id;
  
    m_filetype_copy.filetype_ex.lock();

    clear_filetypes();

    id = filetypes->initEnum();
    ft = (WCFiletype*)filetypes->getFirstElement( id );
    while ( ft != NULL ) {
        m_filetype_copy.filetypes->addElement( ft->duplicate() );
        ft = (WCFiletype*)filetypes->getNextElement( id );
    }
    filetypes->closeEnum( id );
  
    m_filetype_copy.filetype_ex.unlock();
}

void NM_Filetype_Thread::clear_filetypes()
{
    WCFiletype *ft;
    int id;

    m_filetype_copy.filetype_ex.lock();

    id = m_filetype_copy.filetypes->initEnum();
    ft = (WCFiletype*)m_filetype_copy.filetypes->getFirstElement( id );
    while ( ft != NULL ) {
        delete ft;
        m_filetype_copy.filetypes->removeFirstElement();
        ft = (WCFiletype*)m_filetype_copy.filetypes->getFirstElement( id );
    }
    m_filetype_copy.filetypes->closeEnum( id );

    m_filetype_copy.filetype_ex.unlock();
}
