#ifndef INCLUDE_CLASS_BACKLIGHT_H
#define INCLUDE_CLASS_BACKLIGHT_H
/**
 * @brief Class Backlight header file - class definition
 *
 * This files contains the class definition of class backlight.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/class_backlight.h
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#include <pbbtaglist.h>

/**
 * @brief  backlight driver structure
 *
 * This structure contains the public interface to a backlight driver.
 * 
 * The class backlight probes multiple driver until one will correctly
 * initialize. In this case the driver returns a pointer to this
 * structure with will be uses by the class object to call the drivers
 * methods.
 *
 * A new backlight driver must at least define the functions in this
 * structure.
 */
struct driver_backlight {
	/**
	 * @brief  name of this driver
	 *
	 * This string is used to give the user a feedback about used
	 * drivers. It might also be used in the configuration unitlity
	 * PowerPrefs to configure this driver.
	 */
	char *name;
	int  (*get_brightness)();
	int  (*get_brightness_max)();
	void (*set_brightness)();
	void (*driver_exit)();
};

#define INREG(addr)	      readl(memory+addr)
#define OUTREG(addr,val)  writel(val, memory+addr)

static inline unsigned int readl(const volatile void *addr) {
	return *(volatile unsigned int*) addr;
}

static inline void writel(unsigned int b, volatile void *addr) {
	*(volatile unsigned int*) addr = b;
}

/* public prototypes class backlight*/
int backlight_init ();
void backlight_exit ();

/* private prototypes class backlight */
void backlight_query (struct tagitem *taglist);
void backlight_config (struct tagitem *taglist);
void backlight_handle_tags (int cfgure, struct tagitem *taglist);

#endif    /* INCLUDE_CLASS_BACKLIGHT_H */
