/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     bas_fct.c                                                      */
/*                                                                          */
/* description:  collecting information of all Lagrange elements            */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"


/* Macro for a general scalar interpolation function                        */

#define GENERATE_INTERPOL(PREFIX,DEGREE,DIM,N_BAS_FCT)                       \
static const REAL *PREFIX##interpol##DEGREE##_##DIM##d(                      \
                            const EL_INFO *el_info,                          \
			    int no, const int *b_no,                         \
			    REAL (*f)(const REAL_D),                         \
			    REAL (*f_loc)(const EL_INFO *,                   \
					  const REAL [N_LAMBDA]),            \
			    REAL *vec)                                       \
{                                                                            \
  FUNCNAME(#PREFIX "interpol" #DEGREE "_" #DIM "d");                         \
  static REAL       my_vec[N_BAS_FCT];                                       \
  REAL             *rvec = vec ? vec : my_vec;                               \
  int               i;                                                       \
  const PARAMETRIC *parametric = el_info->mesh->parametric;                  \
                                                                             \
  DEBUG_TEST_EXIT(!b_no || (no > 0 && no <= N_BAS_FCT),                      \
		  "not for %d points\n", no);                                \
                                                                             \
  if (f_loc)                                                                 \
    for (i = 0; i < N_BAS_FCT; i++)                                          \
      rvec[i] = f_loc(el_info, PREFIX##bary##DEGREE##_##DIM##d[i]);          \
  else {                                                                     \
    if (parametric) {                                                        \
      REAL_D         world[N_BAS_FCT];                                       \
                                                                             \
      parametric->init_element(el_info, parametric);                         \
      parametric->coord_to_world(el_info, nil, N_BAS_FCT,                    \
				 PREFIX##bary##DEGREE##_##DIM##d, world);    \
                                                                             \
      for (i = 0; i < N_BAS_FCT; i++)                                        \
	rvec[i] = f(world[i]);                                               \
    }                                                                        \
    else {                                                                   \
      REAL_D world;                                                          \
                                                                             \
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);                                 \
                                                                             \
      for (i = 0; i < N_BAS_FCT; i++) {                                      \
	coord_to_world_##DIM##d(el_info, PREFIX##bary##DEGREE##_##DIM##d[i], \
                                world);                                      \
	rvec[i] = f(world);                                                  \
      }                                                                      \
    }                                                                        \
  }                                                                          \
                                                                             \
  if(b_no) { /* Perform resorting if only certain indices are required. */   \
    REAL tmp[N_BAS_FCT];                                                     \
                                                                             \
    memcpy(tmp, rvec, N_BAS_FCT * sizeof(REAL));                             \
                                                                             \
    for(i = 0; i < no; i++)                                                  \
      rvec[i] = tmp[b_no[i]];                                                \
  }                                                                          \
                                                                             \
  return(rvec);                                                              \
}

/* Macro for a general vector interpolation function                        */

#define GENERATE_INTERPOL_D(PREFIX,DEGREE,DIM,N_BAS_FCT)                     \
static const REAL_D *PREFIX##interpol_d##DEGREE##_##DIM##d(                  \
                            const EL_INFO *el_info,                          \
                            int no, const int *b_no,                         \
			    const REAL *(*f)(const REAL_D, REAL_D),          \
 			    const REAL *(*f_loc)(const EL_INFO *,            \
						 const REAL [N_LAMBDA],      \
						 REAL_D),                    \
			    REAL_D *vec)                                     \
{                                                                            \
  FUNCNAME("interpol_d" #DEGREE "_" #DIM "d");                               \
  static REAL_D     my_vec[N_BAS_FCT];                                       \
  REAL_D           *rvec = vec ? vec : my_vec;                               \
  int               i;                                                       \
  const PARAMETRIC *parametric = el_info->mesh->parametric;                  \
                                                                             \
  DEBUG_TEST_EXIT(!b_no || (no > 0 && no <= N_BAS_FCT),                      \
		  "not for %d points\n", no);                                \
                                                                             \
  if (f_loc)                                                                 \
    for (i = 0; i < N_BAS_FCT; i++)                                          \
      f_loc(el_info, PREFIX##bary##DEGREE##_##DIM##d[i], rvec[i]);           \
  else {                                                                     \
    if (parametric) {                                                        \
      REAL_D         world[N_BAS_FCT];                                       \
                                                                             \
      parametric->init_element(el_info, parametric);                         \
      parametric->coord_to_world(el_info, nil, N_BAS_FCT,                    \
				 PREFIX##bary##DEGREE##_##DIM##d, world);    \
                                                                             \
      for (i = 0; i < N_BAS_FCT; i++)                                        \
	f(world[i], rvec[i]);                                                \
    }                                                                        \
    else {                                                                   \
      REAL_D world;                                                          \
                                                                             \
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);                                 \
                                                                             \
      for (i = 0; i < N_BAS_FCT; i++) {                                      \
	coord_to_world(el_info, PREFIX##bary##DEGREE##_##DIM##d[i], world);  \
	f(world, rvec[i]);                                                   \
      }                                                                      \
    }                                                                        \
  }                                                                          \
                                                                             \
  if(b_no) { /* Perform resorting if only certain indices are required. */   \
    REAL_D tmp[N_BAS_FCT];                                                   \
                                                                             \
    memcpy(tmp, rvec, N_BAS_FCT * sizeof(REAL_D));                           \
                                                                             \
    for(i = 0; i < no; i++)                                                  \
      COPY_DOW(tmp[b_no[i]], rvec[i]);                                       \
  }                                                                          \
                                                                             \
  return (const REAL_D *) rvec;                                              \
}

#include "bas_fct_0d.c"
#include "bas_fct_1d.c"
#if DIM_OF_WORLD > 1
#include "bas_fct_2d.c"
#if DIM_OF_WORLD > 2
#include "bas_fct_3d.c"
#endif
#endif

#undef GENERATE_INTERPOL
#undef GENERATE_INTERPOL_D


struct all_bas_fcts
{
  const BAS_FCTS        *bas_fcts;
  struct all_bas_fcts   *next;
};

/*--------------------------------------------------------------------------*/
/*  linked list of all used basis functions: discontinuous Lagrange         */
/*  basisfunctions are always members of the list                           */
/*--------------------------------------------------------------------------*/
#define MAX_DEG 2

static struct all_bas_fcts all_disc_lagrange[DIM_OF_WORLD * (MAX_DEG + 1)] = 
 {{&disc_lagrange0_1d, all_disc_lagrange+1},
  {&disc_lagrange1_1d, all_disc_lagrange+2},
  {&disc_lagrange2_1d,
#if DIM_OF_WORLD > 1
   all_disc_lagrange+3},
  {&disc_lagrange0_2d, all_disc_lagrange+4},
  {&disc_lagrange1_2d, all_disc_lagrange+5},
  {&disc_lagrange2_2d,
#if DIM_OF_WORLD > 2
   all_disc_lagrange+6},
  {&disc_lagrange0_3d, all_disc_lagrange+7},
  {&disc_lagrange1_3d, all_disc_lagrange+8},
  {&disc_lagrange2_3d,
#endif
#endif
 nil}};

const BAS_FCTS *get_discontinuous_lagrange(int dim, int degree)
{
  FUNCNAME("get_discontinuous_lagrange");

  if(dim == 0 || dim > DIM_OF_WORLD) {
    ERROR("Discontinuous Lagrange basis functions of dimension %d are not available for DIM_OF_WORLD == %d!\n", dim, DIM_OF_WORLD);
    return(nil);
  }

  if (degree < 0 || degree > MAX_DEG)
  {
    ERROR("Discontinuous Lagrange basis functions of degree %d are not available\n", degree);
    return(nil);
  }

  return(all_disc_lagrange[(dim - 1) * 3 + degree].bas_fcts);
}

#undef MAX_DEG

/*--------------------------------------------------------------------------*/
/*  linked list of all used basis functions: Lagrange basisfunctions are    */
/*  always members of the list                                              */
/*--------------------------------------------------------------------------*/
#define MAX_DEG 4

static struct all_bas_fcts all_lagrange[1+DIM_OF_WORLD * MAX_DEG] = 
  {{&lagrange_0d, all_lagrange+1},
  {&lagrange1_1d, all_lagrange+2},
  {&lagrange2_1d, all_lagrange+3},
  {&lagrange3_1d, all_lagrange+4},
  {&lagrange4_1d,
#if DIM_OF_WORLD > 1
   all_lagrange+5},
  {&lagrange1_2d, all_lagrange+6},
  {&lagrange2_2d, all_lagrange+7},
  {&lagrange3_2d, all_lagrange+8},
  {&lagrange4_2d,
#if DIM_OF_WORLD > 2
   all_lagrange+9},
  {&lagrange1_3d, all_lagrange+10},
  {&lagrange2_3d, all_lagrange+11},
  {&lagrange3_3d, all_lagrange+12},
  {&lagrange4_3d,
#endif
#endif
   all_disc_lagrange}};

static struct all_bas_fcts *first_bas_fcts = all_lagrange;

const BAS_FCTS *get_lagrange(int dim, int degree)
{
  FUNCNAME("get_lagrange");

  if(dim > DIM_OF_WORLD) {
    ERROR("Lagrange basis functions of dimension %d are not available for DIM_OF_WORLD == %d!\n", dim, DIM_OF_WORLD);
    return(nil);
  }

  if (degree < 1 || degree > MAX_DEG)
  {
    ERROR("no lagrangian basis functions of degree %d\n", degree);
    return(nil);
  }

  if(dim == 0)
    degree = 4;

  return(all_lagrange[(dim - 1) * 4 + degree].bas_fcts);
}

/*--------------------------------------------------------------------------*/
/*  add a set of new basis functions to the list; return true, if possible  */
/*  else false                                                              */
/*--------------------------------------------------------------------------*/

int new_bas_fcts(const BAS_FCTS * bas_fcts)
{
  FUNCNAME("new_bas_fcts");
  struct all_bas_fcts  *new_first;

  if (!bas_fcts)
  {
    ERROR("no basis functions specified; bas_fcts pointer to nil\n");
    return(0);
  }

  TEST_EXIT(bas_fcts->name,
       "new basis functions must have name; bas_fcts->name pointer to nil\n");

  TEST_EXIT(strlen(bas_fcts->name),
	    "new basis functions must have a non empty name\n");
  TEST_EXIT(bas_fcts->dim > 0 && bas_fcts->dim <= 3,
	    "new basis functions must have a dimension between 1 and 3\n");
  TEST_EXIT(bas_fcts->degree >= 0,
	    "new basis functions must have a positive quadrature degree\n");

  TEST_EXIT(bas_fcts->phi,
	    "new basis functions: phi not set\n");
  TEST_EXIT(bas_fcts->grd_phi,
	    "new basis functions: grd_phi not set\n");
  TEST(bas_fcts->D2_phi,
       "Warning: new basis functions: D2_phi not set\n");
  TEST_EXIT(bas_fcts->get_dof_indices,
	    "new basis functions: get_dof_indices not set\n");
  TEST_EXIT(bas_fcts->get_bound,
	    "new basis functions: get_bound not set\n");
  TEST(bas_fcts->interpol,
       "Warning: new basis functions:  interpol not set\n");
  TEST(bas_fcts->interpol_d,
       "Warning: new basis functions: interpol_d not set\n");

  for (new_first = first_bas_fcts; new_first; new_first = new_first->next)
  {
    if (!strcmp(bas_fcts->name, new_first->bas_fcts->name))
    {
      ERROR("basis functions with this name already exist\n");
      TEST_EXIT(bas_fcts == new_first->bas_fcts,
		"pointer to new and existing basis functions differ %p!=%p\n",
		bas_fcts, new_first->bas_fcts);
      ERROR("pointer to new and existing basis functions are the same\n");
      ERROR("ignoring new basis functions; taking old ones\n");
      return(0);
    }
  }

  new_first = MEM_ALLOC(1, struct all_bas_fcts);
  new_first->bas_fcts = bas_fcts;
  new_first->next = first_bas_fcts;
  first_bas_fcts = new_first;
  
  return(1);
}

/*--------------------------------------------------------------------------*/
/*  get a pointer to a set of basis functions from the list; identifier is  */
/*  the name of the basis functions;                                        */
/*  returns a pointer to the BAS_FCTS structure if a corresponding set was  */
/*  found in the list, else pointer to nil                                  */
/*--------------------------------------------------------------------------*/

const BAS_FCTS *get_bas_fcts(const char *name)
{
  FUNCNAME("get_bas_fcts");
  struct all_bas_fcts *bas_fcts;

  if (!name)
  {
    ERROR("no name specified; can not return pointer to basis functions\n");
    return(nil);
  }
  if (!strlen(name))
  {
    ERROR("empty name; can not return pointer to basis functions\n");
    return(nil);
  }
  for (bas_fcts = first_bas_fcts; bas_fcts; bas_fcts = bas_fcts->next)
    if (!strcmp(bas_fcts->bas_fcts->name, name))
      return(bas_fcts->bas_fcts);

  ERROR("basis functions with name %s not found in list of all functions\n");
  return(nil);
}


