/*
 * Copyright (C) 2009 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef WebURLLoaderClient_h
#define WebURLLoaderClient_h

#include "public/platform/WebCommon.h"
#include "public/platform/WebDataConsumerHandle.h"

namespace blink {

class WebURLLoader;
class WebURLRequest;
class WebURLResponse;
struct WebURLError;

class BLINK_PLATFORM_EXPORT WebURLLoaderClient {
 public:
  // Called when following a redirect. |newRequest| contains the request
  // generated by the redirect. The client may modify |newRequest|.
  // |encodedDataLength| is the size of the data that came from the network
  // for this redirect, or zero if the redirect was served from cache.
  //
  // Implementations should return true to instruct the loader to follow the,
  // redirect, or false otherwise.
  virtual bool willFollowRedirect(WebURLLoader*,
                                  WebURLRequest& newRequest,
                                  const WebURLResponse& redirectResponse) {
    return true;
  }

  // Called to report upload progress. The bytes reported correspond to
  // the HTTP message body.
  virtual void didSendData(WebURLLoader*,
                           unsigned long long bytesSent,
                           unsigned long long totalBytesToBeSent) {}

  // Called when response headers are received.
  virtual void didReceiveResponse(WebURLLoader*, const WebURLResponse&) {}

  // Called when response headers are received.
  // The ownership of |handle| is transferred to the callee.
  virtual void didReceiveResponse(WebURLLoader* loader,
                                  const WebURLResponse& response,
                                  WebDataConsumerHandle* handle) {
    delete handle;
    didReceiveResponse(loader, response);
  }

  // Called when a chunk of response data is downloaded. This is only called
  // if WebURLRequest's downloadToFile flag was set to true.
  virtual void didDownloadData(WebURLLoader*,
                               int dataLength,
                               int encodedDataLength) {}

  // Called when a chunk of response data is received. |dataLength| is the
  // number of bytes pointed to by |data|. |encodedDataLength| is the number
  // of bytes actually received from network to serve this chunk, including
  // HTTP headers and framing if relevant. It is 0 if the response was served
  // from cache, and -1 if this information is unavailable.
  // TODO(ricea): -1 is problematic for consumers maintaining a running
  //     total. Investigate using 0 for all unavailable cases.
  // |encodedBodyLength| is the number of bytes used to store this chunk,
  // possibly encrypted, excluding headers or framing. It is set even if the
  // response was served from cache.
  virtual void didReceiveData(WebURLLoader*,
                              const char* data,
                              int dataLength,
                              int encodedDataLength,
                              int encodedBodyLength) {}

  // Called when a chunk of renderer-generated metadata is received from the
  // cache.
  virtual void didReceiveCachedMetadata(WebURLLoader*,
                                        const char* data,
                                        int dataLength) {}

  // Called when the load completes successfully.
  // |totalEncodedDataLength| may be equal to kUnknownEncodedDataLength.
  virtual void didFinishLoading(WebURLLoader* loader,
                                double finishTime,
                                int64_t totalEncodedDataLength) {}

  // Called when the load completes with an error.
  virtual void didFail(WebURLLoader*, const WebURLError&) {}

  // Value passed to didFinishLoading when total encoded data length isn't
  // known.
  static const int64_t kUnknownEncodedDataLength = -1;

 protected:
  virtual ~WebURLLoaderClient() {}
};

}  // namespace blink

#endif
