/*
 * Copyright (C) 2014 Canonical Ltd.
 *
 * This file is part of unity-chromium-extension
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <syslog.h>

#include "connection.h"
#include "inactivity-timer.h"
#include "service.h"
#include "webapps-handler.h"
#include "webapps-process.h"

#include <QCoreApplication>
#include <QDebug>


void syslogOutputHandler(QtMsgType type, const QMessageLogContext &, const QString &msg)
{
    QByteArray localMsg = msg.toLocal8Bit();
    int syslog_type = LOG_USER;
    switch (type) {
    case QtDebugMsg:
        syslog_type |= LOG_DEBUG;
        break;
    case QtWarningMsg:
        syslog_type |= LOG_WARNING;
        break;
    case QtCriticalMsg:
        syslog_type |= LOG_CRIT;
        break;
    case QtFatalMsg:
        syslog_type |= LOG_ERR;
        break;
    }

    syslog(syslog_type, "%s", localMsg.constData());
}


int main(int argc, char **argv)
{
    if (qgetenv("UNITY_WEBAPPS_LOGGING") == QByteArrayLiteral("1")) {
        qInstallMessageHandler(syslogOutputHandler);
    }

    qDebug() << "Starting";

    QCoreApplication app(argc, argv);

    UnityWebapps::Process process;

    UnityWebapps::Service service;
    UnityWebapps::WebappsHandler handler;
    service.addHandler(&handler);

    UnityWebapps::InactivityTimer inactivityTimer(2 * 60 * 1000);
    QObject::connect(&inactivityTimer, SIGNAL(timeout()), &app, SLOT(quit()));
    inactivityTimer.watchObject(&service);

    UnityWebapps::Connection connection;
    QObject::connect(&connection, SIGNAL(messageReceived(const QVariantMap &)),
                     &service, SLOT(handleMessage(const QVariantMap &)));
    QObject::connect(&service, SIGNAL(messageHandled(const QVariantMap &)),
                     &connection, SLOT(postMessage(const QVariantMap &)));
    bool ok = connection.open();
    if (Q_UNLIKELY(!ok)) {
        qCritical() << "Couldn't open connection";
        return EXIT_FAILURE;
    }

    int r = app.exec();

    qDebug() << "Finishing";

    return r;
}

